(**************************************************************)
(*
 *  Ensemble, 1.10
 *  Copyright 2001 Cornell University, Hebrew University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* EVENT.ML *)
(* Author: Mark Hayden, 4/95 *)
(* Based on Horus events by Robbert vanRenesse *)
(**************************************************************)
open Util
open Trans
open Tdefs
(**************************************************************)
let name = Trace.file "EVENT"
let failwith s = Trace.make_failwith name s
(**************************************************************)

type typ =
    (* These events should have messages associated with them. *)
  | ECast				(* Multicast message *)
  | ESend				(* Pt2pt message *)
  | ESubCast				(* Multi-destination message *)
  | ECastUnrel				(* Unreliable multicast message *)
  | ESendUnrel				(* Unreliable pt2pt message *)
  | EMergeRequest			(* Request a merge *)
  | EMergeGranted			(* Grant a merge request *)
  | EOrphan				(* Message was orphaned *)

    (* These types do not have messages. *)
  | EAccount				(* Output accounting information *)
  | EAsync				(* Asynchronous application event *)
  | EBlock				(* Block the group *)
  | EBlockOk				(* Acknowledge blocking of group *)
  | EDump				(* Dump your state (debugging) *)
  | EElect				(* I am now the coordinator *)
  | EExit				(* Disable this stack *)
  | EFail				(* Fail some members *)
  | EGossipExt				(* Gossip message *)
  | EGossipExtDir			(* Gossip message directed at particular address *)
  | EInit				(* First event delivered *)
  | ELeave				(* A member wants to leave *)
  | ELostMessage			(* Member doesn't have a message *)
  | EMergeDenied			(* Deny a merge request *)
  | EMergeFailed			(* Merge request failed *)
  | EMigrate				(* Change my location *)
  | EPresent                            (* Members present in this view *)
  | EPrompt				(* Prompt a new view *)
  | EProtocol				(* Request a protocol switch *)
  | ERekey				(* Request a rekeying of the group *)
  | ERekeyPrcl				(* The rekey protocol events *)
  | ERekeyPrcl2				(*                           *)
  | EStable				(* Deliver stability *)
  | EStableReq				(* Request for stability information *)
  | ESuspect				(* Member is suspected to be faulty *)
  | ESystemError			(* Something serious has happened *)
  | ETimer				(* Request a timer *)
  | EView				(* Notify that a new view is ready *)
  | EXferDone				(* Notify that a state transfer is complete *)
  | ESyncInfo
      (* Ohad, additions *)
  | ESecureMsg				(* Private Secure messaging *)
  | EChannelList			(* passing a list of secure-channels *)
  | EFlowBlock				(* Blocking/unblocking the application for flow control*)

(* Signature/Verification with Auth *)
  | EAuth

  | ESecChannelList                     (* The channel list held by the SECCHAN layer *)
  | ERekeyCleanup
  | ERekeyCommit 

(**************************************************************)

type field =
      (* Common fields *)
  | Type	of typ			(* type of the message*)
  | Peer	of rank			(* rank of sender/destination *)
  | Iov	        of Iovecl.t		(* payload of message *)
  | ApplMsg				(* was this message generated by an appl? *)

      (* Uncommon fields *)
  | Address     of Addr.set		(* new address for a member *)
  | Failures	of bool Arrayf.t	(* failed members *)
  | Presence    of bool Arrayf.t        (* members present in the current view *)
  | Suspects	of bool Arrayf.t        (* suspected members *)
  | SuspectReason of string		(* reasons for suspicion *)
  | Stability	of seqno Arrayf.t	(* stability vector *)
  | NumCasts	of seqno Arrayf.t	(* number of casts seen *)
  | Contact	of Endpt.full * View.id option (* contact for a merge *)
  | HealGos	of Proto.id * View.id * Endpt.full * View.t * Hsys.inet list (* HEAL gossip *)
  | SwitchGos	of Proto.id * View.id * Time.t  (* SWITCH gossip *)
  | ExchangeGos	of string		(* EXCHANGE gossip *)
  | MergeGos	of (Endpt.full * View.id option) * seqno * typ * View.state (* INTER gossip *)
  | ViewState	of View.state		(* state of next view *)
  | ProtoId	of Proto.id		(* protocol id *)
  | Time        of Time.t		(* current time *)
  | Alarm       of Time.t		(* for alarm requests *)
  | ApplCasts   of seqno Arrayf.t 
  | ApplSends   of seqno Arrayf.t
  | DbgName     of string

      (* Flags *)
  | NoTotal				(* message is not totally ordered*)
  | ServerOnly				(* deliver only at servers *)
  | ClientOnly				(* deliver only at clients *)
  | NoVsync
  | ForceVsync
  | Fragment				(* Iovec has been fragmented *)

      (* Debugging *)
  | History	of string		(* debugging history *)

      (* Ohad -- Private Secure Messaging *)
  | SecureMsg of Buf.t
  | ChannelList of (rank * Security.key) list

      (* Ohad -- interaction between Mflow, Pt2ptw, Pt2ptwp and the application*)
  | FlowBlock of rank option * bool

  (* Signature/Verification with Auth *)
  | AuthData of Addr.set * Auth.data

(* Information passing between optimized rekey layers
*)
  | Tree    of bool * Tree.z
  | TreeAct of Tree.sent
  | AgreedKey of Security.key

  | SecChannelList of Trans.rank list  (* The channel list held by the SECCHAN layer *)
  | SecStat of int                    (* PERF figures for SECCHAN layer *)
  | RekeyFlag of bool                 (* Do a cleanup or not *)

(**************************************************************)

type t = {
  typ	 : typ ;
  peer   : rank ;
  iov	 : Iovecl.t ;
  applmsg : bool ;
  extend : field list
}

type up = t
type dn = t

(**************************************************************)

type dir = Up of t | Dn of t

type ('a,'b) dirm = UpM of t * 'a | DnM of t * 'b

(**************************************************************)

let invalid_rank = -1

(**************************************************************)

let bodyCore debug typ peer iov = {
  typ 	  = typ ;
  peer	  = peer ;
  applmsg = false ;
  iov	  = iov ;
  extend  = []
}

(**************************************************************)

let castIov debug iov = {
  typ 	  = ECast ;
  peer    = invalid_rank ;
  applmsg = false ;
  iov	  = iov ;
  extend  = []
}

(**************************************************************)

let castIovAppl debug iov = {
  typ 	  = ECast ;
  peer    = invalid_rank ;
  applmsg = true ;
  iov	  = iov ;
  extend  = []
}

(**************************************************************)

let sendPeerIovAppl debug rank iov = {
  typ 	  = ESend ;
  peer 	  = rank ;
  applmsg = true ;
  iov	  = iov ;
  extend  = []
}

(**************************************************************)

let sendPeerIov debug rank iov = {
  typ 	  = ESend ;
  peer 	  = rank ;
  applmsg = false ;
  iov	  = iov ;
  extend  = []
}

(**************************************************************)

let rec setLoop iov pee app typ ext usl = match usl with
  | [] -> {peer=pee;typ=typ;iov=iov;applmsg=app;extend=ext}
  | hd :: tl -> match hd with
    | Type   typ -> setLoop iov pee app typ ext tl
    | Peer   pee -> setLoop iov pee app typ ext tl
    | Iov    iov -> setLoop iov pee app typ ext tl
    | ApplMsg    -> setLoop iov pee true typ ext tl
    | f          -> setLoop iov pee app typ (f::ext) tl

let set debug {typ=typ;iov=iov;peer=pee;applmsg=app;extend=ext} dsl =
  setLoop iov pee app typ ext dsl

let setIovFragment debug ev iov = {ev with iov=iov;extend=Fragment::ev.extend}
let setNoTotal debug ev = {ev with extend=NoTotal::ev.extend}
let setPeer debug ev peer = {ev with peer=peer}
let setSendUnrelPeer debug ev peer = {ev with typ=ESendUnrel;peer=peer}

(**************************************************************)

let create debug typ fields =
  setLoop Iovecl.empty invalid_rank false typ [] fields

(*
let free debug ev = Iovecl.free (Refcnt.info name debug) ev.iov
let copy debug ev = Iovecl.ref (Refcnt.info name debug) ev.iov ; ev
*)
let free debug ev = Iovecl.free debug ev.iov
let free_noIov = ignore2
let copy debug ev = { ev with iov = Iovecl.copy debug ev.iov }

let upCheck debug ev = Iovecl.check debug ev.iov
let dnCheck debug ev = Iovecl.check debug ev.iov

(**************************************************************)

let getExtender f g =
  let rec loop l =
    match l with
    | [] -> g
    | h::t ->
	match f h with
	| Some o -> o
	| None -> loop t
  in fun e -> loop e.extend

let getExtendFail f field =
  let rec loop l =
    match l with
    | [] -> failwith (sprintf "get%s:no such field" field)
    | h::t ->
	match f h with
	| Some o -> o
	| None -> loop t
  in fun e -> loop e.extend

let getExtendOpt e f =
  let rec loop l = 
    match l with
    | [] -> ()
    | h::t ->
     	if not (f h) then loop t
  in loop e.extend

let getExtend  e = e.extend
let getIov     e = e.iov
let getPeer    e = e.peer
let getType    e = e.typ
let getApplMsg e = e.applmsg

let getDbgName       = getExtendFail (function DbgName i       -> Some i | _ -> None) "DbgName"
let getAlarm         = getExtendFail (function Alarm i         -> Some i | _ -> None) "Alarm"
let getContact       = getExtendFail (function Contact(i,j)    -> Some(i,j) | _ -> None) "Contact"
let getFailures      = getExtendFail (function Failures i      -> Some i | _ -> None) "Failures"
let getPresence      = getExtendFail (function Presence b      -> Some b | _ -> None) "Presence"
let getAddress       = getExtendFail (function Address i       -> Some i | _ -> None) "Address"
let getNumCasts      = getExtendFail (function NumCasts i      -> Some i | _ -> None) "NumCasts"
let getProtoId       = getExtendFail (function ProtoId i       -> Some i | _ -> None) "ProtoId"
let getStability     = getExtendFail (function Stability i     -> Some i | _ -> None) "Stability"
let getSuspectReason = getExtendFail (function SuspectReason i -> Some i | _ -> None) "SuspectReason"
let getSuspects      = getExtendFail (function Suspects i      -> Some i | _ -> None) "Suspects"
let getTime          = getExtendFail (function Time i          -> Some i | _ -> None) "Time"
let getViewState     = getExtendFail (function ViewState l     -> Some l | _ -> None) "ViewState"
let getApplCasts     = getExtendFail (function ApplCasts l     -> Some l | _ -> None) "ApplCasts"
let getApplSends     = getExtendFail (function ApplSends l     -> Some l | _ -> None) "ApplSends"

let getSecureMsg     = getExtendFail (function (SecureMsg m) -> Some m | _ -> None) "SecureMsg"
let getChannelList   = getExtendFail (function (ChannelList l) -> Some l | _ -> None) "ChannelList"
let getFlowBlock    = getExtendFail (function (FlowBlock (ro,b)) -> Some ((ro,b)) | _ -> None) "FlowBlock"
let getClientOnly ev = match ev.extend with [] -> false | ext -> List.memq ClientOnly ext
let getServerOnly ev = match ev.extend with [] -> false | ext -> List.memq ServerOnly ext
let getNoTotal ev    = match ev.extend with [] -> false | ext -> List.memq NoTotal    ext
let getNoVsync ev    = match ev.extend with [] -> false | ext -> List.memq NoVsync    ext
let getForceVsync ev = match ev.extend with [] -> false | ext -> List.memq ForceVsync ext
let getFragment ev   = match ev.extend with [] -> false | ext -> List.memq Fragment   ext
let getAuthData      = getExtendFail (function (AuthData (a,data)) -> Some (a,data) | _ -> None) "AuthData" 
let getSecChannelList = getExtendFail (function (SecChannelList l) -> Some l | _ -> None) "SecChannelList"
let getSecStat = getExtendFail (function (SecStat i) -> Some i | _ -> None) "SecStat"
let getTree = getExtendFail (function Tree (a,b) -> Some (a,b) | _ -> None) "Tree"
let getTreeAct = getExtendFail (function TreeAct a -> Some a | _ -> None) "TreeAct"
let getAgreedKey = getExtendFail (function (AgreedKey a) -> Some a | _ -> None) "AgreedKey"
let getRekeyFlag = getExtendFail  (function RekeyFlag flg -> Some flg | _ -> None) "RekeyFlag"
(**************************************************************)

let getIovLen ev =
  let iov = getIov ev in
  Iovecl.len name iov

(**************************************************************)

let string_of_type = Trace.debug "" (function
  | EAccount	    	-> "EAccount"
  | EAsync	  	-> "EAsync"
  | EBlock	  	-> "EBlock"
  | EBlockOk	  	-> "EBlockOk"
  | ECast	  	-> "ECast"
  | ECastUnrel	  	-> "ECastUnrel"
  | EDump	  	-> "EDump"
  | EElect		-> "EElect"
  | EExit	  	-> "EExit"
  | EFail	  	-> "EFail"
  | EGossipExt		-> "EGossipExt"
  | EGossipExtDir	-> "EGossipExtDir"
  | EInit	  	-> "EInit"
  | ELeave	    	-> "ELeave"
  | ELostMessage 	-> "ELostMessage"
  | EMergeRequest 	-> "EMergeRequest"
  | EMergeDenied 	-> "EMergeDenied"
  | EMergeFailed 	-> "EMergeFailed"
  | EMergeGranted  	-> "EMergeGranted"
  | EMigrate            -> "EMigrate"
  | EOrphan	  	-> "EOrphan"
  | EPresent            -> "EPresent"
  | EPrompt             -> "EPrompt"
  | EProtocol		-> "EProtocol"
  | ERekey              -> "ERekey"
  | ERekeyPrcl          -> "ERekeyPrcl"
  | ERekeyPrcl2         -> "ERekeyPrcl2"
  | ESend	  	-> "ESend"
  | ESendUnrel	  	-> "ESendUnrel"
  | EStable	  	-> "EStable"
  | EStableReq	  	-> "EStableReq"
  | ESubCast            -> "ESubCast"
  | ESuspect	  	-> "ESuspect"
  | ESystemError 	-> "ESystemError"
  | ETimer	  	-> "ETimer"
  | EView	  	-> "EView"
  | EXferDone		-> "EXferDone"
  | ESyncInfo           -> "ESyncInfo"
  | ESecureMsg          -> "ESecureMsg"
  | EChannelList        -> "EChannelList"
  | EFlowBlock          -> "EFlowBlock"
  | EAuth               -> "EAuth"
  | ESecChannelList     -> "ESecChannelList"
  | ERekeyCleanup       -> "ERekeyCleanup"
  | ERekeyCommit        -> "ERekeyCommit"
)

(**************************************************************)

let string_of_field f g name v = if f v then Some(name^"="^(g v)) else None
let int_field_neg   = string_of_field (fun i -> i >= 0) string_of_int
let bool_field_false = string_of_field ident string_of_bool
let int_list_field  = string_of_field (fun v -> v <> []) string_of_int_list
let int_array_field = string_of_field (fun v -> v <> [||]) string_of_int_array

(**************************************************************)

let string_of_fields = Trace.debug "" (function
  | Failures	e -> ("Failures="^(Arrayf.bool_to_string e))
  | DbgName     e -> ("DbgName="^e)
  | Presence    e -> ("Presence="^(Arrayf.bool_to_string e))
  | Suspects	e -> ("Suspects="^(Arrayf.bool_to_string e))
  | SuspectReason e -> ("SuspectReason="^(e))
  | Stability	e -> ("Stability="^(Arrayf.int_to_string e))
  | NumCasts	e -> ("NumCasts="^(Arrayf.int_to_string e))
  | ApplCasts	e -> ("ApplCasts="^(Arrayf.int_to_string e))
  | ApplSends	e -> ("ApplSends="^(Arrayf.int_to_string e))
  | Contact	(e,f) -> ("Contact="^(Endpt.string_of_full e))
  | HealGos	(e,f,g,h,i) -> ("HealGos")
  | SwitchGos	(e,f,g) -> ("SwitchGos")
  | ExchangeGos	e -> ("ExchangeGos")
  | MergeGos	(e,f,g,h) -> ("MergeGos")
  | History	e -> ("History="^(e))
  | ViewState	e -> ("ViewState="^(View.string_of_state e))
  | ProtoId	e -> ("ProtoId="^(Proto.string_of_id e))
  | Time        e -> ("Time="^(Time.to_string e))
  | Alarm       e -> ("Alarm="^(Time.to_string e))
  | ApplMsg       -> "ApplMsg"
  | NoTotal       -> "NoTotal"
  | ClientOnly    -> "ClientOnly"
  | ServerOnly    -> "ServerOnly"
  | SecureMsg   m -> "SecureMsg(_)" 
  | ChannelList l -> "ChannelList ["^(string_of_list (function (r,key) -> 
		  string_of_int r ^":"^ Security.string_of_key key) l) ^"]"
  | FlowBlock  (ro,b) -> "FlowBlock" ^ 
                         (match ro with 
			    | Some(r) -> string_of_int r
			    | None -> "multicast"
			 ) ^
                         (string_of_bool b)
  | AuthData (a,data) -> "EPrivateEnc"
  | Tree        _ -> "Tree _"
  | TreeAct     _ -> "TreeAct _"
  | AgreedKey   _ -> "AgreedKey"
  | Address     _ -> "Address(_)"
  | _ -> failwith "unknown field"
)

let to_string = Trace.debug "" (fun ev ->
  let l = filter_nones [
    int_field_neg "peer" ev.peer ;
    bool_field_false "appl" ev.applmsg
  ] @ (List.map string_of_fields ev.extend) in
  sprintf "%s{%s}" 
    (string_of_type ev.typ) 
    (String.concat ":" l)
)

(**************************************************************)

let castEv	  debug	     = create debug ECast[]
let castUnrel	  debug	     = create debug ECastUnrel[]
let castPeerIov   debug o i  = create debug ECast[Peer o;Iov i]
let sendPeer      debug d    = create debug ESend[Peer d]
let sendUnrelPeer debug d    = create debug ESendUnrel[Peer d]
let suspectReason debug s r  = create debug ESuspect[Suspects s;SuspectReason r]
let timerAlarm    debug t    = create debug ETimer[Alarm t]
let timerTime     debug t    = create debug ETimer[Time t]
let dumpEv        debug      = create debug EDump[]

(**************************************************************)
