(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* UTIL.ML *)
(* Author: Mark Hayden, 4/95 *)
(**************************************************************)

external (=|) : int -> int -> bool = "%eq"
external (<>|) : int -> int -> bool = "%noteq"
external (>=|) : int -> int -> bool = "%geint"
external (<=|) : int -> int -> bool = "%leint"
external (>|) : int -> int -> bool = "%gtint"
external (<|) : int -> int -> bool = "%ltint"

(**************************************************************)

let verbose = ref false
let ident x = x
let zero = 0.0

(**************************************************************)

let fprintf ch fmt =
  let f s = 
    output_string ch s ;
    flush ch
  in
  Printe.f f fmt

let sprintf fmt = Printe.f ident fmt
let printf  fmt = fprintf stdout fmt
let eprintf fmt = fprintf stderr fmt

(**************************************************************)

let optimizations = Hashtbl.create 10

let register_opt name v =
  Hashtbl.add optimizations name v

let print_opts () =
  eprintf "OPTIMIZATIONS:\n" ;
  Hashtbl.iter (fun name v ->
    eprintf "  %s : %s\n" name v
  ) optimizations

(**************************************************************)

let make_failwith s1 s2 = failwith (s1^":"^s2) ; ()
let make_assert f p s = if not p then f s
let failmsg a b = (a^":"^b)

(**************************************************************)

let if_some o f = 
  match o with
  | None -> ()
  | Some a -> f a

let some_of = function
  | None -> failwith "some_of"
  | Some a -> a

let is_none = function
  | None -> true
  | Some _ -> false

let string_of_option sov = function
  | None -> "None"
  | Some v -> sprintf "Some(%s)" (sov v)

let option_map f = function
  | None -> None
  | Some o -> Some(f o)
      
let filter_nones l =
  List.fold_right (fun a b -> match a with
    | None -> b
    | Some a -> a :: b
  ) l []			    

let once l =
  match (filter_nones l) with
  | [] -> failwith "once:never"
  | [a] -> a
  | _ -> failwith "once:multiple Some's"

(**************************************************************)

let hex_of_string s =
  let n = String.length s in
  let h = String.create (2 * n) in
  for i = 0 to pred n do
    let c = s.[i] in
    let c = Char.code c in
    let c = sprintf "%02X" c in
    String.blit c 0 h (2 * i) 2
  done ;
  h

(**************************************************************)

let counter () =
  let count = ref (-1) in
  fun () ->
    incr count ;
    !count

(**************************************************************)
(* SEQUENCE: [sequence n] evaluates to the list [0;1;...;n-1].
 *)
let sequence n =
  let rec loop i l =
    if i >= 0 then loop (pred i) (i::l)
    else l
  in loop (n-1) []

(**************************************************************)

let for_array f v =
  for i = 0 to (Array.length v) - 1 do
    f i (v.(i))
  done

(**************************************************************)
(* TransisE addition
 *)

let array_exists f v = 
  let r = ref false in 
  for i = 0 to pred (Array.length v) do
    if f i (v.(i)) then
      r := true
  done ;
  !r
  
(**************************************************************)

let array_create name l i =
(*
  if l > 10 then 
    eprintf "UTIL:array_creat:%s, len=%d\n" name l ;
*)
  Array.create l i

let array_createf n f =
  if n = 0 then [||] else (
    let fst = f 0 in
    let a = array_create "array_createf" n fst in
    for i = 1 to pred n do
      a.(i) <- f i
    done ;
    a
  )

(**************************************************************)

let array_filter f a =
  let b = Array.copy a in
  let j = ref 0 in
  Array.iter (fun it ->
    if f it then (
      b.(!j) <- it ;
      incr j
    )
  ) a ;
  Array.sub b 0 !j

(**************************************************************)

let array_filter_nones a =
  let a = array_filter (fun it -> it <> None) a in
  let a = Array.map some_of a in
  a

(**************************************************************)

let array_fold_left f accu ar =
  let len = Array.length ar in
  let rec loop i accu =
    if i < len then
      loop (succ i) (f accu (ar.(i)))
    else accu
  in loop 0 accu

(**************************************************************)

let array_combine a b =
  let l = Array.length a in
  if l <> Array.length b then
    failwith "array_combine" ;
  if l = 0 then [||] else
    array_createf l (fun i -> (a.(i),b.(i)))

let array_split c =
  let a = Array.map fst c in
  let b = Array.map snd c in
  (a,b)

(**************************************************************)

let array_index it v =
  let rec loop i =
    if i >= Array.length v then
      raise Not_found
    else if v.(i) = it then
      i
    else loop (i+1)
  in loop 0

(**************************************************************)

let array_mem it v =
  let rec loop i =
    if i < 0 then false
    else if v.(i) = it then true
    else loop (i-1)
  in loop ((Array.length v) - 1)

(**************************************************************)

let array_for_all f a =
  let rec loop i =
    if i < 0 then true
    else if f a.(i) then loop (pred i)
    else false
  in loop (pred (Array.length a))

(**************************************************************)

let array_for_all2 f a1 a2 =
  if Array.length a1 <> Array.length a2 then
    failwith "array_for_all2:mismatched arrays" ;
  let rec loop i =
    if i < 0 then true
    else if f a1.(i) a2.(i) then loop (pred i)
    else false
  in loop (pred (Array.length a1))

(**************************************************************)

let array_incr v i = v.(i) <- succ v.(i)
let array_add v i j = v.(i) <- v.(i) + j
let array_sub v i j = v.(i) <- v.(i) - j
let matrix_incr v i j = v.(i).(j) <- succ v.(i).(j)

(**************************************************************)

let queue_clean f q =
  try while true do
    f (Queue.take q)
  done with Queue.Empty -> ()

(**************************************************************)

let list_of_queue q =
  let l = ref [] in
  Queue.iter (fun it ->
    l := it :: !l
  ) q ;
  List.rev !l

(**************************************************************)

let list_filter f =
  List.fold_left (fun l i ->
    if f i then i :: l else l
  ) []

(**************************************************************)

let hashtbl_size t =
  let n = ref 0 in
  Hashtbl.iter (fun _ _ -> incr n) t ;
  !n

(**************************************************************)

let list_of_hashtbl h =
  let l = ref [] in
  Hashtbl.iter (fun k d ->
    l := (k,d) :: !l
  ) h ;
  !l

(**************************************************************)

let hashtbl_clean h =
  try
    Hashtbl.iter (fun k _ ->
      Hashtbl.remove h k
    ) h
  with Not_found -> 
    failwith "hashtbl_clean:Not_found"

(**************************************************************)

let just_once f =
  let cache = ref None in
  fun () ->
    match !cache with
    | None ->
      	let v = f () in
      	cache := Some v ;
	v
    | Some v -> v

(**************************************************************)

let string_map f s =
  let len = String.length s in
  let t = String.create len in
  for i = 0 to pred (String.length s) do
    t.[i] <- f s.[i]
  done;
  t

(**************************************************************)

let strchr c s =
  let n = String.length s in
  let rec loop i =
    if i >= n then 
      raise Not_found
    else if s.[i] = c then
      i
    else loop (succ i)
  in loop 0

(**************************************************************)

let strstr c s =
  let n = String.length s in
  let rec loop i =
    if i >= n then 
      raise Not_found
    else if List.mem s.[i] c then
      i
    else loop (succ i)
  in loop 0

let chars_of_string s =
  let l = ref [] in
  for i = 0 to pred (String.length s) do
    l := s.[i] :: !l
  done ;
  !l

(**************************************************************)

let string_split c s =
  let c = chars_of_string c in
  let rec loop s =
    try
      let i = strstr c s in
      let hd = String.sub s 0 i in
      let tl = String.sub s (i+1) (String.length s - i - 1) in
      hd::(loop tl)
    with Not_found -> [s]
  in loop s

(**************************************************************)

let bool_of_string s =
  let s = String.uppercase s in
  match s with
  | "T" | "TRUE" -> true
  | "F" | "FALSE" -> false
  | _ -> failwith "bool_of_string:bad argument"

(**************************************************************)

let string_of_list f l   = sprintf "[%s]" (String.concat "|" (List.map f l))
let string_of_pair fa fb = fun (a,b) -> sprintf "(%s,%s)" (fa a) (fb b)
let string_of_array f v  = string_of_list f (Array.to_list v)
let string_of_int_array  = string_of_array string_of_int
let string_of_int_list   = string_of_list string_of_int
let string_of_bool       = function true -> "t" | false -> "f"
let string_of_bool_list  = string_of_list string_of_bool
let string_of_bool_array = string_of_array string_of_bool

(**************************************************************)
(**************************************************************)

let except el =
  let rec loop = function
    | [] -> []
    | hd::tl -> if el = hd then tl else hd::loop tl
  in loop
  
let index a =
  let rec loop i = function
    | []  -> raise Not_found
    | b::l -> if a = b then i else loop (succ i) l
  in loop 0
  
(**************************************************************)

let word_len = 4
let mask1 = pred word_len
let mask2 = lnot mask1

let is_aligned i = i land mask1 = 0
let floor_word i = i land mask2
let ceil_word i = (i + mask1) land mask2

(**************************************************************)
(* Adapted from Objective Caml Printexc module.
 *)

let string_of_exn = function
| Out_of_memory ->
    sprintf "Out of memory"
| Match_failure(file, first_char, last_char) ->
    sprintf "Match_failure(%s,%d,%d)" file first_char last_char
| x ->
    let name = Obj.magic(Obj.field (Obj.field (Obj.repr x) 0) 0) in
    let fields = ref [] in
    if Obj.size (Obj.repr x) > 1 then (
      for i = 1 to Obj.size (Obj.repr x) - 1 do
        if i > 1 then prerr_string ", ";
        let arg = Obj.field (Obj.repr x) i in
        if not (Obj.is_block arg) then
          fields := string_of_int (Obj.magic arg : int) :: !fields
        else if Obj.tag arg = 252 then begin
          fields := sprintf "'%s'" (Obj.magic arg : string) :: !fields
        end else
          fields := "_" :: !fields
      done;
      sprintf "%s(%s)" name (String.concat "," !fields)
    ) else (
      name
    )

(**************************************************************)

let make_magic () = 
  ((Obj.repr,Obj.magic) : ('a -> Obj.t) * (Obj.t -> 'a))

let make_marsh_buf name catch_unmarsh =
  let marsh obj buf ofs len =
    Hsys.marshal (Obj.repr obj) buf ofs len

  and unmarsh buf ofs len =
    let (obj,pos) =
      try Obj.unmarshal buf ofs with Failure(s) as e when catch_unmarsh ->
      	eprintf "UTIL:unmarsh:%s:%s\n" name (string_of_exn e) ;
	eprintf "  len=%d, avail_len=%d\n" len (String.length buf - ofs) ;

	(* Print out magic number information.
	 *)
	let tmp = Obj.marshal (Obj.repr ()) in
    	let intext_magic_number = Hsys.pop_int tmp 0 in
	let magic = Hsys.pop_int buf ofs in
	eprintf "  magic number: expected=%08x got=%08x\n" intext_magic_number magic ;

	exit 1
    in 
    
    (* Check that unmarshaller did not read too much.
     *)
    if ceil_word pos <>| ofs + len then (
      eprintf "UTIL:pos=%d, ofs=%d, len=%d\n" pos ofs len ;
      failwith "UTIL:unmarsh:bad message len" ;
    ) ;
    Obj.magic obj
  in
(*
  let marsh obj buf ofs len =
    let ret = Hsys.marshal (Obj.repr obj) buf ofs len in
    printf "UTIL:marsh:%d bytes\n" ret ;
    ret
  in
*)

  (* BUG: performance *)
(*
  let unmarsh buf ofs len =
    let check = String.copy buf in
    let obj = unmarsh buf ofs len in
    if buf <> check then
      failwith "marshaller modified buffer" ;
    obj
  in
*)

(*
  let unmarsh buf ofs len =
    unmarsh (String.sub buf ofs len) 0 len
  in
*)
(*
  (* BUG: performance *)
  let marsh o buf ofs len =
    let len = marsh o buf ofs len in
    if len >=| 0 then (
      let s = String.sub buf ofs len in
      let no = unmarsh s 0 len in
      if o <> no then 
	failwith "bad marshalled object"
    ) ;
    len
  in
*)

  (marsh,unmarsh)

let make_marsh name catch_unmarsh =
  let marshal,unmarshal = make_marsh_buf name catch_unmarsh in
  
  let marshal obj =
    let rec loop len =
      let buf = String.create len in
      let ret = marshal obj buf 0 len in
      if ret <| 0 then 
      	loop (4 * len)
      else
      	String.sub buf 0 ret
    in loop (63*4)
  in

  (marshal,unmarshal)

(* Turn on the debugging...*)
(*
let make_marsh name =
  let id = Digest.string name in
  let problem s = failwith (sprintf "problem:marsh:%s:%s" name s) in

  let marsh o = 
    let str = Obj.marshal (Obj.repr o) in
    let len = String.length str in
    let d   = Digest.string str in
    let l   = String.create 4 in
    Hsys.push_int l 0 len ;
    String.concat "" [l;id;d;str]
  in
  let unmarsh buf ofs len =
    if ofs < 0 ||  ofs + len > String.length buf then
      problem "unmarsh:short[a]" ;
    let str = String.sub buf ofs len in

    if String.length str < 20 then
      problem (sprintf "unmarsh:short(len=%d)[b]" (String.length str)) ;
    let id' = String.sub str 4 16 in
    if id <> id' then (
      eprintf "UTIL:unmarsh:id = %s\n" (hex_of_string id) ;
      eprintf "UTIL:unmarsh:id'= %s\n" (hex_of_string id') ;
      problem "unmarsh:id mismatch"
    ) ;

    let len = Hsys.pop_int str 0 in 
    if len + 36 > (String.length str) then
      problem "unmarsh:short" ;

    let  d' = String.sub str 20 16 in
    let d = Digest.substring str 36 len in
    if d <> d' then problem "unmarsh:signature mismatch" ;

    let (o,_) = Obj.unmarshal (String.sub str 36 len) 0 in
    let o = Obj.magic o in
    o
  in 
  (marsh,unmarsh)
*)
(**************************************************************)
(* Support for keeping track of message sizes *)

let average =
  let cntr = ref 0
  and sum  = ref 0
  and min_len  = ref 10000 in
  fun i ->
    incr cntr ;
    sum := !sum + i ;
    if i < !min_len then min_len := i ;
    if (!cntr mod 1000) = 0 then (
      eprintf "MSG:accounting:#msgs=%d, total bytes=%d, min=%d avg=%.2f\n"
      	!cntr !sum !min_len (float !sum /. float !cntr)
    )

(**************************************************************)

let string_of_inet =
  let hosts = Hashtbl.create 10 in
  fun inet ->
    try Hashtbl.find hosts inet
    with Not_found ->
      let name = Hsys.string_of_inet inet in
      Hashtbl.add hosts inet name ;
      name

(**************************************************************)

let disable_sigpipe =
  just_once (fun () ->
    try
      if !verbose then
      	eprintf "UTIL:warning:disabling SIGPIPE signals\n" ;
      Sys.signal Sys.sigpipe Sys.Signal_ignore
    with _ -> ()
  )

(**************************************************************)

let gc_profile name f arg =
  let s1 = Gc.stat () in
  let res = f arg in
  let s2 = Gc.stat () in
  let words = s1.Gc.minor_words - s2.Gc.minor_words in
  printf "GC_PROFILE:%s:%d words\n" name words ;
  res

let gc_profile3 name f = 
  fun a1 a2 a3 ->
  let s1 = Gc.stat () in
  let res = f a1 a2 a3 in
  let s2 = Gc.stat () in
  let words = s2.Gc.minor_words - s1.Gc.minor_words in
  printf "GC_PROFILE:%s:%d words\n" name words ;
  res

(**************************************************************)
(* TransisE additions
*)
let some_less x y =
  match x with
  | None -> (match y with
    | None -> false
    | Some _ -> true)
  | Some x_val -> (match y with
    | None -> false
    | Some y_val -> x_val < y_val)

let some_inc x = 
  match x with
  | None -> 0
  | Some z -> z+1

let some_min x y = 
  match x with
  | None -> None
  | Some x_val -> (match y with
    | None -> None
    | Some y_val -> Some (min x_val y_val))

let some_max x y = 
  match x with
  | None -> (match y with
    | None -> None
    | Some y_val -> Some y_val)
  | Some x_val -> (match y with
    | None -> Some x_val
    | Some y_val -> Some (max x_val y_val))

let some_comp x y = 
  match x with
  | None -> (match y with
    | None -> true
    | Some y_val -> false)
  | Some x_val -> (match y with
    | None -> false
    | Some y_val -> if x_val=y_val then true
	else false)

let print_some x = (
  match x with
  | None -> Printf.printf "N|"
  | Some z -> Printf.printf "%d|" z
)

let int_of_some x = 
  match x with
  | None -> -1
  | Some z -> z

let max_vct x y = 
  for i=0 to (Array.length x)-1 do 
    x.(i) <- some_max x.(i) y.(i) 
  done

(**************************************************************)

let choose a k =
  let n = Array.length a in
  let k = min n k in
  let rec loop k l =
    if k = 0 then l else (
      let i = a.(Random.int n) in
      if List.mem i l then
      	loop k l
      else
      	loop (pred k) (i::l)
    )
  in loop k []

let gossip failed rank =
  let l = ref [] in
  for i = 0 to pred (Array.length failed) do
    if (not failed.(i)) && i <> rank then 
      l := i :: !l
  done ;
  Array.of_list !l

let random_list l =
  let l = Array.of_list l in
  let n = Array.length l in
  for i = 1 to pred n do
    let j = Random.int i in 
    let tmp = l.(i) in
    l.(i) <- l.(j) ;
    l.(j) <- tmp
  done ;
  Array.to_list l
 
(**************************************************************)

let chars_of_string s =
  let l = ref [] in
  for i = 0 to pred (String.length s) do
    l := s.[i] :: !l
  done ;
  !l

let string_of_char_list =
  string_of_list Char.escaped

let str_verb = ref false

let strip_prefix s c =
  if !str_verb then
    eprintf "strip_prefix:%s:%s\n" 
      s (string_of_char_list c) ;
  let res =
    let rec loop i =
      if i >= String.length s then ""
      else if List.mem s.[i] c then loop (succ i)
      else String.sub s i ((String.length s) - i)
    in loop 0
  in
  if !str_verb then
    eprintf "strip_prefix:%s\n" res ;
  res

let strtok s c =
  let c = chars_of_string c in
  if !str_verb then
    eprintf "strtok:%s:%s\n" 
      s (string_of_char_list c) ;
  let s = strip_prefix s c in
  let res =
    let rec loop i =
      if String.length s = 0 then 
      	raise Not_found
      else if i >= String.length s then
      	(s,"")
      else if List.mem s.[i] c then
	let tok = String.sub s 0 i in
	let s = String.sub s i ((String.length s) - i) in
	let s = strip_prefix s c in
	(tok,s)
      else loop (succ i)
    in loop 0
  in
  if !str_verb then
    eprintf "strtok:out:%s:%s\n" (fst res) (snd res) ;
  res

(**************************************************************)

let make_map debug id mapping =
  let rmapping = List.map (fun (a,b) -> (b,a)) mapping in
  let id_of_string s = 
    let t = String.uppercase s in
    try List.assoc t mapping with Not_found ->
      failwith (sprintf "%s:%s_of_string:no such id:'%s'" debug id s)
  in
  
  let string_of_id a =
    try 
      let r = List.assoc a rmapping in
      String.capitalize (String.lowercase r)
    with Not_found ->
      failwith (sprintf "%s:string_of_%s:bad id" debug id)
  in
  id_of_string, string_of_id

(**************************************************************)

let read_lines () =
  let stdin = Hsys.stdin () in
  let buf = String.create 10000 in
  let len = Hsys.read stdin buf 0 (String.length buf) in
  if len = 0 then raise End_of_file ;
  let buf = String.sub buf 0 len in
  let rec loop buf =
    if buf = "" then [] else (
      let tok,buf = strtok buf "\n" in
      if tok = "" then (
	loop buf 
      ) else (
	tok :: loop buf
      )
     )
  in
  loop buf

(**************************************************************)

let addinfo s t = 
  s^":"^t

(**************************************************************)

let string_list_of_gc_stat s =
  let alloc_promote_pct = (s.Gc.promoted_words * 100) / s.Gc.minor_words in
  let alloc_major_direct = s.Gc.major_words - s.Gc.promoted_words in
  let blocks_total = s.Gc.live_blocks + s.Gc.free_blocks in
  let blocks_live_pct = (s.Gc.live_blocks * 100) / blocks_total in
  let words_live_pct = (s.Gc.live_words * 100) / s.Gc.heap_words in
  [ sprintf "allocation: minor=%dM (%d%% promoted) (direct major=%dK)" 
    (s.Gc.minor_words/1000000) alloc_promote_pct (alloc_major_direct/1000) ;
    sprintf "collections: minor=%d, major=%d" s.Gc.minor_collections s.Gc.major_collections ;
    sprintf "words: %d (%d%% live) (%d chunks)" s.Gc.heap_words words_live_pct s.Gc.heap_chunks ;
          sprintf "blocks: %d (%d%% live) (largest_free=%d)" blocks_total blocks_live_pct s.Gc.largest_free
  ]	

(**************************************************************)
