(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* TRACE.ML *)
(* Author: Mark Hayden, 3/96 *)
(**************************************************************)
open Util
(**************************************************************)

let callbacks = ref ([] : (unit -> unit) list)

let install cb =
  callbacks := cb :: !callbacks

(**************************************************************)

let source_report = ref (try Sys.getenv "ENS_TRACE" ; true with _ -> false)

let sources = ref []

let source_set_report v =
  source_report := v

let source_file file =
  sources := file :: !sources ;
  if !source_report then (
    eprintf "TRACE:entering:%s\n" file ;	
    flush stderr
  ) ;
  file

(**************************************************************)
(**************************************************************)
let configs = ref []

let config key data =
  configs := (key,data) :: !configs

(**************************************************************)
(**************************************************************)
let tests = ref []

let declare_test name f =
  tests := (name,f) :: !tests

(**************************************************************)

let exec_test name =
  eprintf "TRACE:executing test '%s'\n" name ; flush stdout ;
  try
    let f = List.assoc name !tests in
    Printexc.catch f () ;
    exit 0
  with Not_found -> (
    eprintf "TRACE:couldn't find test '%s'\n" name ;
    exit 0
  )

(**************************************************************)
(**************************************************************)

let print_config () =
  eprintf "TRACE:config_print:begin\n" ;
  eprintf "  Source Files (that reported)\n" ;
  List.iter (fun file -> 
    eprintf "    %s\n" file) (List.rev !sources) ;
  eprintf "  Config Info\n" ;
  List.iter (fun (key,data) -> 
    eprintf "    %s=%s\n" key data) !configs ;
  eprintf "  Internal Tests\n" ;
  List.iter (fun (name,_) -> 
    eprintf "    %s\n" name) !tests ;
  List.iter (fun cb -> cb ()) !callbacks ;
  eprintf "TRACE:config_print:end\n" ;
  flush stdout

(**************************************************************)

let roots = Queue.create ()

let install_root f = Queue.add f roots

let print_roots () = 
  printf "\n" ;
  Queue.iter (fun f -> 
    let l = f () in
    List.iter (fun line ->
      printf "%s\n" line
    ) l
  ) roots ;
  printf "\n" ;
  flush stdout

(**************************************************************)

let logs = Hashtbl.create 10

let get_log name =
  let name = String.uppercase name in
  try Hashtbl.find logs name with Not_found ->
    let ret = ref None in
    Hashtbl.add logs name ret ;
    ret

let log name info =
  let name = String.uppercase name in
  let x = get_log name in
  fun g ->
    match !x with
    | Some f -> f info (g ())
    | None -> ()

let logl name info =
  let name = String.uppercase name in
  let x = get_log name in
  fun g ->
    match !x with
    | Some f -> List.iter (f info) (g())
    | None -> ()

let add_log name f =
  let name = String.uppercase name in
  let x = get_log name in
  x := Some(f)

let rmv_log name =
  let name = String.uppercase name in
  let x = get_log name in
  x := None

let list_of_logs () =
  let l = list_of_hashtbl logs in
  let l = List.map fst l in
  l

(**************************************************************)

let comments = Queue.create ()

let comment s = Queue.add s comments

let print_comments () =
  eprintf "Embedded comments:\n" ;
  Queue.iter (fun s -> eprintf "  %s\n" s) comments

(**************************************************************)

let _ = install_root (fun () ->
    [sprintf "TRACE:#logs=%d, #comments=%d" 
      (hashtbl_size logs) (Queue.length comments)]
  )

(**************************************************************)
