(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* SCHED.ML: call-back scheduling *)
(* Author: Mark Hayden, 12/95 *)
(**************************************************************)
open Printf
(**************************************************************)

(* Count of # of scheduling steps so far.
 *)
let steps = ref 0

(* ITEM: Always visible
 *)
type item =
| Arg0 of (unit -> unit)
| Arg1 of (Obj.t -> unit) *
    Obj.t
| Arg2 of (Obj.t -> Obj.t -> unit) *
    Obj.t * Obj.t
| Arg3 of (Obj.t -> Obj.t -> Obj.t -> unit) *
    Obj.t * Obj.t * Obj.t
| Arg4 of (Obj.t -> Obj.t -> Obj.t -> Obj.t -> unit) *
    Obj.t * Obj.t * Obj.t * Obj.t


(* TYPED_ITEM: Always visible
 *)
type ('a,'b,'c,'d) typed_item =
| T_Arg0 of (unit -> unit)
| T_Arg1 of ('a -> unit) 			* 'a
| T_Arg2 of ('a -> 'b -> unit) 			* 'a * 'b
| T_Arg3 of ('a -> 'b -> 'c -> unit) 		* 'a * 'b * 'c
| T_Arg4 of ('a -> 'b -> 'c -> 'd -> unit) 	* 'a * 'b * 'c * 'd

(**************************************************************)
(*
type t = (unit -> unit) Queuee.t

let new _ 	= Queuee.create ()
and on _ 	= ()
and off _ 	= ()
and check _ 	= ()
and enqueue s f =
  Queuee.add f s
and enqueue_1arg s f a =
  Queuee.add (fun () -> f a) s
and enqueue_2arg s f a b =
  Queuee.add (fun () -> f a b) s
and enqueue_3arg s f a b c =
  Queuee.add (fun () -> f a b c) s
and enqueue_4arg s f a b c d =
  Queuee.add (fun () -> f a b c d) s
and empty = queue_empty
and size = Queuee.length
and step s n =
  try
    for i = 1 to n do
      (Queuee.take s) ()
    done ;
    true
  with Queuee.Empty -> false
*)
(**************************************************************)

type t = item Queuee.t

let create _ 	= Queuee.create ()
and on _ 	= ()
and off _ 	= ()
and check _ 	= ()
and enqueue s f =
  Queuee.add (Arg0(f)) s
and enqueue_1arg s f a =
  Queuee.add (Obj.magic (Obj.repr (T_Arg1(f,a)))) s
and enqueue_2arg s f a b =
  Queuee.add (Obj.magic (Obj.repr (T_Arg2(f,a,b)))) s
and enqueue_3arg s f a b c =
  Queuee.add (Obj.magic (Obj.repr (T_Arg3(f,a,b,c)))) s
and enqueue_4arg s f a b c d =
  Queuee.add (Obj.magic (Obj.repr (T_Arg4(f,a,b,c,d)))) s
and empty = Queuee.empty
and size = Queuee.length

and step s =
  let rec loop n =
    if n <= 0 then
      true
    else if Queuee.empty s then
      false
    else (
      begin
	match Queuee.take s with
	| Arg0(f) 	  -> f ()
	| Arg1(f,a) 	  -> f a
	| Arg2(f,a,b) 	  -> f a b
	| Arg3(f,a,b,c)   -> f a b c
	| Arg4(f,a,b,c,d) -> f a b c d
      end ;
      incr steps ;			(* PERF *)
      loop (pred n)
    )
  in loop

(**************************************************************)
(*
type t = {
  q 		: item Queuee.t ;
  mutable on 	: int  ;
  name 		: string
}

let create name = {
    q 	 = Queuee.create () ;
    on   = 0 ;
    name = name
  }
and on s = s.on <- succ s.on
and off s = s.on <- pred s.on
and check s =
  if s.on < 0 then (
    printf "SCHED:name=%s\n" s.name ;
    failwith "SCHED:op when off"
  )

let enqueue s f            = check s ; Queuee.add (Arg0(f)) s.q
and enqueue_1arg s f a     = check s ; Queuee.add (Arg1(Obj.magic (Obj.repr f), (Obj.repr a))) s.q
and enqueue_2arg s f a b   = check s ; Queuee.add (Arg2((Obj.magic (Obj.repr f)), (Obj.repr a), (Obj.repr b))) s.q
and enqueue_3arg s f a b c = check s ; Queuee.add (Arg3((Obj.magic (Obj.repr f)), (Obj.repr a), (Obj.repr b), (Obj.repr c))) s.q
and enqueue_4arg s f a b c d = check s ; Queuee.add (Arg4((Obj.magic (Obj.repr f)), (Obj.repr a), (Obj.repr b), (Obj.repr c), (Obj.repr c))) s.q
and empty s  = Queuee.empty s.q
and size s   = Queuee.length s.q
and step s n =
  try
    for i = 1 to n do
      match Queuee.take s.q with
      | Arg0(f) -> f ()
      | Arg1(f,a) -> f a
      | Arg2(f,a,b) -> f a b
      | Arg3(f,a,b,c) -> f a b c
      | Arg4(f,a,b,c,d) -> f a b c d
    done ; true
  with Queue.Empty -> false
*)
(**************************************************************)
