(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* REFCNT.ML *)
(* Author: Mark Hayden, 3/95 *)
(**************************************************************)
let name = Trace.source_file "REFCNT"
let failwith s = failwith (Util.failmsg name s)
let log = Trace.log name ""
(**************************************************************)
open Util
(**************************************************************)
let actually_use_counts = ref false
let force_major_gc = ref true
(**************************************************************)
(* GORPGORP
module Weak = struct
  type 'a t = 'a option array
  let create l = array_create name l None
  let set a i o = Array.set a i o
  let get a i = Array.get a i
end
*)
(**************************************************************)
(**************************************************************)

type 'a t = {
  mutable count : int ;
  obj : 'a ;
  mutable debug : (string * string) list
} 

(**************************************************************)

let finalized obj =
  let h = Weak.create 1 in

  (* This next step has to allocate an object.
   *)
  let o = {
    count = 1 ;
    debug = [] ; 
    obj = obj
  } in
  Weak.set h 0 (Some o) ;
  (h,o)

let is_live s =
  match Weak.get s 0 with
  | None -> false
  | Some handle -> 
      (not !actually_use_counts) || (handle.count > 0)

(**************************************************************)

(* If we use a Stack the resident set size should be small,
 * though with a Queue, things should be safer.
 *)
module Freelist = Stack
let fl_add = Freelist.push
let fl_take = Freelist.pop

(**************************************************************)

type nlive = int
type nfree = int

type 'a pool = {
  alloc : unit -> 'a ;
  free : 'a -> unit ;
  make : nlive -> nfree -> 'a list ;
  fl : 'a Freelist.t ;
  mutable alive : ('a t Weak.t * 'a) list
}    

(**************************************************************)

let pool name alloc free make = {
  alloc = alloc ;
  free = free ;
  make = make ;
  fl = Freelist.create () ;
  alive = []
}

(**************************************************************)
(* Scan through live buffers looking for freed buffers to
 * add to the free list.  
 *)

let clean p =
  let recovered = ref 0 in
  let rec loop = function
    | [] -> []
    | (handle,obj) as hd :: tl ->
	if is_live handle then (
	  hd :: (loop tl)
        ) else (
	  incr recovered ;
	  fl_add obj p.fl ;
	  loop tl 
	)
  in
  p.alive <- loop p.alive ;
  log (fun () -> sprintf "recovered %d buffers" !recovered)

(**************************************************************)

let use_gc p =
  (* If still empty, then:
   *)
  let nlive = List.length p.alive in
  log (fun () -> sprintf "Gc.full_major: total alive=%d" nlive) ;
  Gc.full_major () ;		(* Do a full GC *)
  clean p ;			(* And grab our buffers *)
  
  (* Figure out current state and call the policy function.
   *)
  let live = List.length p.alive in
  let free = Freelist.length p.fl in
  let l = p.make live free in
  log (fun () -> sprintf "alloced %d" (List.length l)) ;
  
  (* Ensure that there will be at least one buffer in
   * the free list.  
   *)
  if free = 0 && l = [] then 
    failwith "bad policy function" ;
  
  (* Add the new buffers and take one out.
   *)
  List.iter (fun o -> fl_add o p.fl) l ;
  try fl_take p.fl
  with Freelist.Empty -> 
    failwith "sanity"

(**************************************************************)

let alloc debug p =
  let obj =
  (* First check the free list.
   *)
    try fl_take p.fl
    with Freelist.Empty -> 
      (* If empty, then collect any freed buffers.
       *)
      clean p ;
      try fl_take p.fl
      with Freelist.Empty -> 
	if !force_major_gc then (
	  use_gc p
	) else (
	  p.alloc()
	)
  in
  let weak_ptr,weak_obj = finalized obj in
  p.alive <- (weak_ptr,obj) :: p.alive ;
  weak_obj

(**************************************************************)

let void debug obj = {
  count = 1000000 ;
  debug = ["void","*void*"] ;
  obj = obj
}

let check debug h = 
()(*  
  if h.count < 0 then
    eprintf "REFCNT:warning:negative refcount:rc=%d:%s\n" 
      h.count debug
  *)

let incr debug h = 
  check debug h ;
  h.count <- succ h.count

let decr debug h = 
  h.count <- pred h.count 
(*;
  check debug h
*)

(*
let read debug h f =
  check debug h ;
  f h.obj
*)

let read debug h = h.obj

let pool_debug p = 
  let l =
    List.map (fun (ww,_) ->
      match Weak.get ww 0 with
      |	Some h ->
	  if h.count > 0 then (
	    let debug = (string_of_list (fun (a,b) -> sprintf "%s=%s" a b) h.debug) in
	    Some(sprintf "count=%d %s" h.count debug)
	  ) else None
      |	None -> None
    ) p.alive
  in filter_nones l
  
(**************************************************************)
(**************************************************************)
(*
let counter = Util.counter ()

let alloc d p =
  let (h,o) = alloc d p in
  let debug = sprintf "alloc(%d)" (counter ()) in
  h.debug <- (debug,d) :: h.debug ;
  (h,o)

let check d h = 
  if h.count < 0 then (
    eprintf "REFCNT:dropped below zero\n" ;
    eprintf "  %s\n" (string_of_list (fun (a,b) -> sprintf "%s=%s" a b) h.debug) ;
    exit 0
  )

let incr d h =
  h.debug <- ("incr",d) :: h.debug ;
  incr d h ;
  check d h
  
let decr d h =
  h.debug <- ("decr",d) :: h.debug ;
  decr d h ;
  check d h

let check d h =
  h.debug <- ("check",d) :: h.debug ;
  check d h

*)
(**************************************************************)
(**************************************************************)
