(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* PRIQ.ML: heap-based priority queues *)
(* Author: Mark Hayden, 3/96 *)
(**************************************************************)
open Util
(**************************************************************)
let name = "PRIQ"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type ('a,'b) item = {
  mutable time : 'a ;
  mutable item : 'b option
}

type ('a,'b) t = {
  zero			: 'a ;
  mutable min		: 'a ;
  mutable alarms	: ('a,'b) item array ;
  mutable nalarms	: int
}

(**************************************************************)

let new_array n zero =
  let v = array_createf n (fun _ -> {time=zero;item=None}) in
  if v.(1) == v.(2) then failwith "sanity";
  v

let create zero = {
  zero		= zero ;
  min		= zero ;
  alarms	= new_array 30 zero ;
  nalarms	= 0
}

let size a = a.nalarms
let min a = a.min
let empty a = a.nalarms = 0

let to_list t =
  let a = Array.sub t.alarms 1 t.nalarms in
  let a = Array.map (fun i -> (i.time,some_of i.item)) a in
  Array.to_list a

let clear t f =
  for i = 1 to t.nalarms do
    let it = t.alarms.(i) in
    begin
      match it.item with
      | None -> failwith "sanity"
      | Some item ->
	  f it.time item
    end ;
    it.time <- t.zero ;
    it.item <- None ;
  done ;
  t.nalarms <- 0

(**************************************************************)

let set a i it =
(* commented for performance.
  if i = 0 then
    failwith "setting zeroth" ;
  if i > a.nalarms then
    failwith "setting unused" ;
*)
  a.alarms.(i) <- it

let add a t item =
  a.nalarms <- succ a.nalarms ;
  if a.nalarms >= Array.length a.alarms then (
    let na = new_array (2 * Array.length a.alarms) a.zero in
    for i = 0 to pred (Array.length a.alarms) do
      na.(i) <- a.alarms.(i)
    done ;
    a.alarms <- na
  ) ;

  let rec loop i =
    let j = i / 2 in
    if j <= 0 || t >= a.alarms.(j).time then (
      set a i {time=t;item=Some(item)} ;
(*
      let it = a.alarms.(i) in
      it.time <- t ;
      it.item <- Some(item) ;
*)
      if j = 0 then
	a.min <- t
    ) else (
      set a i a.alarms.(j) ;
      loop j
    )
  in loop a.nalarms

let take a =
  if a.nalarms <= 0 then failwith "sanity" ;
  let alarms = a.alarms in

  let old_min = alarms.(1) in
  let item = some_of old_min.item in

  let rec loop i =
    let l = 2 * i in
    let r = succ l in
    if r <= a.nalarms then (
      let lt = alarms.(l).time in
      let rt = alarms.(r).time in
      if lt < rt then (
	set a i alarms.(l) ;
	loop l
      ) else (
	set a i alarms.(r) ;
	loop r
      )
    ) else if l > a.nalarms then (
      i
    ) else if r > a.nalarms then (
      if l <> a.nalarms then failwith "sanity" ;
      set a i alarms.(l) ;
      loop l
    ) else failwith "sanity"
  in

  let k = loop 1 in

  set a k alarms.(a.nalarms) ;

  let swap = alarms.(k) in
  let rec loop i =
    let j = i / 2 in
    if j <= 0 || swap.time >= alarms.(j).time then (
      set a i swap
    ) else (
      set a i alarms.(j) ;
      loop j
    )
  in loop k ;

  set a a.nalarms {time=a.zero;item=None} ;
(*
  let it = alarms.(a.nalarms) in
  it.time <- zero ;
  it.item <- None ;
*)
  a.nalarms <- pred a.nalarms ;

  a.min <- alarms.(1).time ;
  item

let get a f t =
  if t < a.min || a.nalarms =| 0 then
    false
  else (
    while a.min <= t & a.nalarms >| 0 do
      let time = a.min in
      f time (take a)
    done ;
    true
  )

let getopt a t =
  if t < a.min || a.nalarms =| 0 then
    false
  else (
    while a.min <= t && a.nalarms >| 0 do
      (take a) t
    done ;
    true
  )

(**************************************************************)
(*
open Printf

let print a =
  printf "ALARM(%3.2f,%d)" a.min a.nalarms ;
  for i = 1 to a.nalarms do
    printf ":%3.2f" a.alarms.(i).time
  done ;
  printf "\n" ;
  

let check a =
  if a.nalarms > 0 & a.min <> a.alarms.(1).time then
    failwith "min not set" ;
  if a.alarms.(0) <> {time=zero;item=None} then
    failwith "zeroth got modified" ;
  for i = succ a.nalarms to pred (Array.length a.alarms) do
    if a.alarms.(i) <> {time=zero;item=None} then (
      printf "size=%d, i=%d\n" a.nalarms i ;
      failwith "unused got modified"
    )
  done ;

(*
  for i = 1 to a.nalarms do
    match a.alarms.(i) with
    | {time=t1;item=Some(t2)} -> (
	if t1 <> t2 then failwith "bad vals"
      )
    | _ -> failwith "sanity"
  done ;
*)

  for i = 2 to a.nalarms do
    let j = i / 2 in
    if a.alarms.(j).time > a.alarms.(i).time then (
      printf "predicate broke: %d,%3.2f > %d,%3.2f\n"
	j a.alarms.(j).time
	i a.alarms.(i).time ;
      print a ;
      failwith "predicate broke"
    )
  done

let test () =
(*
  let a = new () in

  let n = 18 in

  for i = 1 to n do
    check a ;
    let t = Random.float 100.0 in
    add a t t
  done ;

  for i = 1 to n do
    check a ;
    print a ;
    take a
  done ;
  check a ;
  print a ;

  for i = 1 to 100 do
    check a ;
    let t = Random.float 1000.0 in
    add a t t
  done ;

  for i = 1 to 100 do
    check a ;
    let t = float (i * 10) in
    printf "%3.2f\n" t ;
    get a t (fun t -> printf "	%3.2f\n" t)
  done ;

  check a ;
  print a ;
*)
(*
  let a = new () in
  let l = ref [] in

  for i = 1 to 1000000 do
    if (i mod 100) = 0 then (
      printf "TIMER:test(%d), size=%d\n" i (size a)
    ) ;

    let cmp a b = compare a b >= 0 in
    let l1 = Sort.list cmp !l in
    let l2 = Sort.list cmp (list_of_timer a) in
(*
    List.iter (fun t -> printf "1:%f\n" t) l1 ;
    List.iter (fun t -> printf "2:%f\n" t) l2 ;
*)
    if l1 <> l2 then (
      printf "TIMER:bad intersect (i=%d)\n" i ;
      failwith "sanity"
    ) ;
    check a ;
    match Random.int 7 with
    | 0 | 1 | 2 -> (
	let t = Random.float 1.0 in
	printf "added:%3.2f\n" t ;
	add a t t ;	
	l := t :: !l ;
	()
      )	
    | _ -> (
	if size a > 0 then (
	  let t = take a in
	  printf "took:%3.2f\n" t ;
	  l := except t !l
	)
      )
  done
*)
(Trace.declare_test "timer" test)
*)
(**************************************************************)
(* Functorized version. 
 * Todo: roll some of the stuff into here.
 *)

module type OrderedType =
  sig
    type t
    val zero : t
    val ge : t -> t -> bool
  end
