(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* MCREDIT.ML *)
(* Author: Mark Hayden, Zhen Xiao, 3/97 *)
(**************************************************************)
open Trans
open Util
(**************************************************************)
let name = "MCREDIT"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type credit = int

type t = {
    my_rank : rank ;
    nmembers : nmembers ;
    ack_thresh : credit ;
    send_credit : credit array ;
    recv_credit : credit array ;
    mutable msg_len : int ;
    ack_que : credit Queuee.t array ;
    mutable waitfor : rank ;
    failed : bool array ;
    mutable all_failed : bool
  } 

(* Note: we require that initially send_credit satisfies
 * send_credit.(i) < send_credit.(i+1) for i = 0 to n-2
 *)
let check rank ack_thresh send_credit =
  let ok = ref true in
  let nmembers = Array.length send_credit in
(* Problem here with skipping my own entry.

  for i = 0 to nmembers - 2 do
    if i <> rank 
    && send_credit.(i) >= send_credit.(succ i) 
    then (
      eprintf "MCREDIT:send_credit=%s, rank=%d\n" 
        (string_of_int_array send_credit) rank ;
      failwith "send_credit not increasing"
    )
  done ;
*)
  if ack_thresh < send_credit.(nmembers-1) - send_credit.(0) then
    failwith "ack threshhold too small"

let create my_rank nmembers ack_thresh send_credit recv_credit = 
  check my_rank ack_thresh send_credit ;
  let waitfor =
    if nmembers = 1 || my_rank <> 0 then 0 else 1
  in {
    my_rank = my_rank ;
    nmembers = nmembers ;
    ack_thresh = ack_thresh ;
    send_credit = send_credit ;
    recv_credit = recv_credit ;
    msg_len = 0 ;
    ack_que = array_createf nmembers (fun _ -> Queuee.create ()) ;
    waitfor = waitfor ;
    failed = array_create name nmembers false ;
    all_failed = (nmembers = 1)
  } 


(* Note: wait_next would go into indefinite loop if all members in
   the group have failed.
 *)
let increase m =
  if m.all_failed then 
    failwith "increase:but all members are failed" ;
  let rec wait_next m =
    m.waitfor <- (succ m.waitfor) mod m.nmembers ;
    if m.my_rank =| m.waitfor 
    || m.failed.(m.waitfor) 
    then 
      wait_next m
  in

  wait_next m ;
  if m.waitfor < 0 || m.waitfor >= m.nmembers then
    failwith "sanity" ;

  while not (Queuee.empty m.ack_que.(m.waitfor)) do
    let credit = Queuee.take m.ack_que.(m.waitfor) in 
    array_add m.send_credit m.waitfor credit ;
    wait_next m
  done

(* Note: we require credit > send_credit.(n-1) - send_credit.(0) for
   the initial values of send_credit. This way after any member gets a
   credit message, its credit becomes the greatest in the group.
 *)
let got_credit m origin =
  if m.failed.(origin) then 
    failwith "got_credit from failed member" ;
  if m.waitfor = origin then (
    array_add m.send_credit m.waitfor m.ack_thresh ;
    increase m
  ) else (
    Queuee.add m.ack_thresh m.ack_que.(origin)
  )
	  

let got_msg m origin msg_len =
  array_add m.recv_credit origin msg_len ;
  m.recv_credit.(origin)

let set_credit m origin remainder =
  m.recv_credit.(origin) <- remainder

let fail m rank =
  if rank = m.my_rank then
    failwith "I'm dead?" ;
  m.failed.(rank) <- true ;
  m.send_credit.(rank) <- 0 ;
  m.recv_credit.(rank) <- 0 ;
  Queuee.clear m.ack_que.(rank) ;

  let alive = ref false in
  for i = 0 to pred m.nmembers do
    if i <> m.my_rank && (not m.failed.(i)) then
      alive := true
  done ;
  m.all_failed <- not !alive ;

  if (not m.all_failed) && m.waitfor = rank then 
    increase m

let check m =
  m.send_credit.(m.waitfor) > m.msg_len

(* We may want to change the name "take" since now we actually add
 * something.
 *)

let take m credit =
  m.msg_len <- m.msg_len + credit


(* Note the complexity of this function is not constant. However,
 * since it is called only periodically, this may not be a problem.
 *)

let clear m =
  let credit = min m.msg_len m.send_credit.(m.waitfor) in
  for rank = 0 to pred m.nmembers do
    if not m.failed.(rank) then
      array_sub m.send_credit rank credit ;
  done ;
  m.msg_len <- m.msg_len - credit


let string_of_queue_len q = string_of_int (Queuee.length q)

let to_string_list m = [
  sprintf "my_rank = %d  nmembers = %d  msg_len = %d"
    m.my_rank m.nmembers m.msg_len ;
  sprintf "send_credit = %s" 
    (string_of_int_array m.send_credit) ;
  sprintf "ack_que = %s"
    (string_of_array string_of_queue_len m.ack_que) ;
  sprintf "recv_credit = %s"
    (string_of_int_array m.recv_credit)
]  

