(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* MBUF.MLI *)
(* Author: Mark Hayden, 3/95 *)
(**************************************************************)
open Trans
(**************************************************************)

type t

(* A global mbuf for everyone to use.
 *)
val global : t

(* Create a new mbuf.
 *)
val create : 
  debug ->				(* name of mbuf *)
  len ->				(* size of segments *)
  len ->				(* max alloc amount *)
  t

(* Get the maximum allocation size of an mbuf.
 *)
val max_len : t -> len

(**************************************************************)
(* Various allocation functions for mbufs. *)

(* Allocate space and copy data into an mbuf.
 *)
val alloc : debug -> t -> buf -> ofs -> len -> Iovec.t

(* Allows reads into mbufs to be made dynamically.  The read
 * function is given a pointer into the current mbuf (which
 * it can read up to the mac alloc amount).  The returned
 * length is used to update the pointer into the mbuf.  And
 * the next function is then called with the data returned by
 * the read.
 *)
val alloc_dyn : 
  debug -> t ->
  (buf -> ofs -> len -> len) ->		(* read: get the message *)
  (Iovec.t -> unit) ->			(* alloced: buffer has been alloced *)
  (unit -> unit) ->			(* error: called when error is returned *)
  (bool -> bool)			(* check for message *)

(* Allocate from an mbuf and allow the recieve function
 * to return a value.
 *)
val alloc_fun : 
  debug -> t -> 
  (buf -> ofs -> len -> len * 'a) -> 
  Iovec.t * 'a

(* Perhaps misnamed fast, this is yet another way to allocate
 * off of an mbuf.
 *)
val fast :
  debug -> t -> 
  (len -> (int -> len -> unit) -> 'gorp) -> (* preprocessor *)
  ('gorp -> buf -> ofs -> bool) ->	(* poller *)
  (int -> buf Refcnt.t -> ofs -> len -> unit) ->(* handler *)
  (unit -> unit) ->			(* error *)
  (unit -> bool)			(* result *)

(**************************************************************)
(* Marshaller to iovecs and from strings.  When marshalling,
 * tries to allocate from the given mbuf.
 *)
val make_marsh :
  debug -> t -> (('a -> Iovec.t) * (buf -> ofs -> len -> 'a))

(* Flatten an iovecl using the given mbuf.  If the resulting
 * iovec is small enough to be allocated with the mbuf, then
 * it is allocated there.  On the ML heap otherwise.
 *)
val flatten :
  debug -> t -> Iovecl.t -> Iovec.t

(**************************************************************)
val pool : t -> buf Refcnt.pool
(**************************************************************)
