(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* MBUF.ML *)
(* Author: Mark Hayden, 3/95 *)
(**************************************************************)
open Util
open Trans
(**************************************************************)
let name = Trace.source_file "MBUF"
let failwith s = failwith (Util.failmsg name s)
let log = Trace.log name ""
(**************************************************************)

(* GORPGORP
let mbuf_size = Hsys.max_msg_len()+1		(* 256K *)
*)
let mbuf_size = 256 * 1024		(* 256K *)
let mbuf_count = ref 1			(* # bufs allocated *)

let _ =
  Trace.install_root (fun () ->
    [sprintf "MBUF:#mbufs=%d" (!mbuf_count)]
  )

(**************************************************************)

type t = {
  debug : string ;
  max_len : int ;
  hi : ofs ;
  mutable rbuf : Iovec.rbuf ;
  mutable ofs : ofs ;
  pool : buf Refcnt.pool
}

let max_len m = m.max_len
let pool m = m.pool

(**************************************************************)

(* This is the number of free buffers we expect to have after
 * calling Gc.full_major ().  If we do not get that many back,
 * then allocate at least that many.
 *)
let free_const = 6

let buffer_policy live free =
  if live = 0 && free = 0 then (
    (* We must not be using the special buffer management system.
     *)
    1
  ) else (
    let add = 0 in
(* I think this is bad...
    let add = max add (live - free) in
*)
    let add = max add (free_const - free) in
    let add = min add free_const in
    log (fun () -> sprintf "live=%d, free=%d, adding=%d" live free add) ;
    add
  )

(* String create, tries to create a string on the heap.
 * BUG: this code should be linked to the call to finalize
 *)
let alloc len =
  try 
    Hsys.static_string len
  with Failure _ ->
    String.create len

let free o =
  try 
    Hsys.static_string_free o
  with Failure _ -> ()

let allocer len live free =
  let add = buffer_policy live free in
  let add = array_create name add () in
  let add = Array.map (fun () -> alloc len) add in
  let add = Array.to_list add in
  add

(**************************************************************)

let refresh m f =
  incr mbuf_count ;
  let rbuf = m.rbuf in
  m.rbuf <- Refcnt.alloc name m.pool ;
  m.ofs <- 0 ;
  f () ;
  Refcnt.decr name m.rbuf

let advance m len =
  let len = ceil_word len in
  m.ofs <- m.ofs + len ;
  if m.ofs >| m.hi then  (
    refresh m ident
  )

(**************************************************************)

let create debug len max_len = 
  if len <=| max_len then
    failwith "create: len <= max_len" ;
  let pool = 
    Refcnt.pool
      debug
      (fun () -> alloc len) 
      free
      (allocer len) 
  in

  { rbuf = Refcnt.alloc debug pool ;
    ofs = 0 ;
    hi = len - max_len ;
    max_len = max_len ;
    debug = debug ;
    pool = pool
  }

(**************************************************************)

let global =
  Printexc.catch (fun () -> 
    create "global" mbuf_size (Hsys.max_msg_len ())
  ) ()

(**************************************************************)

let chk_aligned i =
  if i land 3 <> 0 then
    failwith "alignment error"

(**************************************************************)

let alloc debug m buf ofs len =
  if ceil_word len <>| len then
    failwith "alloc:non-aligned length" ;
  if len >| m.max_len then 
    failwith "alloc:len>max_len" ;

  let m_buf = Refcnt.read debug m.rbuf in
  String.blit buf ofs m_buf m.ofs len ;

  let ret = Iovec.alloc debug m.rbuf m.ofs len in
  advance m len ;
  ret

(**************************************************************)

let alloc_dyn debug m recv hdlr error =
  let () = () in			(* HACK (?) *)

  (* Critical path.
   *)
  fun rm ->
    (* Call the receive function.
     *)
    let len = recv (Refcnt.read debug m.rbuf) m.ofs m.max_len in

    if len >| 0 then (
      let iov = Iovec.alloc debug m.rbuf m.ofs len in
      advance m len ;
      hdlr iov ;
      true
    ) else if len <| 0 then (
      error () ; rm
    ) else rm

(**************************************************************)

let make_marsh debug m =
  let debug = addinfo "MBUF(mm)" debug in
  let (marsh,unmarsh) = Util.make_marsh_buf debug true in
  let (marsh_bad,_) = Util.make_marsh debug true in
  let marsh obj =
    let len =
      marsh obj (Refcnt.read debug m.rbuf) m.ofs m.max_len
    in

    if len >=| 0 then (
      let ret = Iovec.alloc debug m.rbuf m.ofs len in
      advance m len ;
      ret
    ) else (
      let str = marsh_bad obj in
      Iovec.alloc name (Iovec.heap debug str) 0 (String.length str)
    )      
  in (marsh,unmarsh)

(**************************************************************)

let flatten debug m = 
  let debug = addinfo "MBUF(flat)" debug in
  fun il ->
    match Array.length il with
    | 0 -> Iovec.empty name
    | 1 ->
	Iovec.ref debug il.(0) ;
	il.(0)
    | _ ->
	let len = Iovecl.len debug il in
	if len >=| m.max_len then (
	  Iovecl.flatten debug il
	) else (
	  Iovecl.flatten_buf debug il (Refcnt.read debug m.rbuf) m.ofs m.max_len ;
	  let iov = Iovec.alloc debug m.rbuf m.ofs len in
	  advance m len ;
	  iov
	)

(**************************************************************)

let fast debug m pre poll recv err =
  let close dint len =
    let buf = m.rbuf in
    let ofs = m.ofs in

    (* Round len up and add to offset.
     *)
    let clen = ceil_word len in
    m.ofs <- m.ofs + clen ;
    if m.ofs >| m.hi then (
      refresh m (fun () -> recv dint buf ofs len)
    ) else (
      recv dint buf ofs len
    )
  in

  let gorp = pre m.max_len close in
  
  fun rm -> 
    poll gorp (Refcnt.read debug m.rbuf) m.ofs (*|| rm*)

(**************************************************************)
(**************************************************************)
(* This is taken from cdda/mllib/cdmbuf.ml by Jason Hickey.
 *)

(* More functional version of alloc_dyn.
 * The read function is given a pointer into the current mbuf
 * (which it can read up to the max alloc amount).  The returned
 * length is used to update the pointer into the mbuf, and the "other"
 * value is returned directly.
 *)
let alloc_fun debug m recv =
  (* Call the receive function.
   *)
  let len, other =
    recv (Refcnt.read debug m.rbuf) m.ofs m.max_len
  in
  
  (* Special case for zero length iovec.
   *)
  if len =| 0 then
    Iovec.empty debug, other
  else (
    (* Perhaps we should do something about zero length iovecs *)
    let iov = Iovec.alloc debug m.rbuf m.ofs len in
     
    (* Round len up and add to offset.
     *)
    let len = ceil_word len in
    advance m len ;
    
    iov, other
  )

(**************************************************************)
(**************************************************************)
