(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* MARSH.ML *)
(* Author: Mark Hayden, 12/96 *)
(**************************************************************)

exception Error of string

let failwith s = raise (Error ("MARSH:"^s))

(**************************************************************)

type marsh = string list ref

let init () = ref []
let add m v = m := v :: !m

let write_int m i =
  let n = String.create 4 in
  Hsys.push_int n 0 i ;
  add m n

let write_bool m b =
  if b then 
    write_int m 1
  else 
    write_int m 0

let write_string m s =
  let l = String.length s in
  write_int m l ;
  add m s ;

  begin
    match l mod 4 with
    | 0 -> ()
    | 1 -> add m (String.create 3)
    | 2 -> add m (String.create 2)
    | 3 -> add m (String.create 1)
    | _ -> failwith "sanity"
  end

let write_list m f l =
  let len = List.length l in
  write_int m len ;
  List.iter (fun it -> f it) l

let write_option m f o =
  match o with
  | None -> write_bool m false
  | Some o -> 
      write_bool m true ;
      f o

let marsh m =
  let m = List.rev !m in
  String.concat "" m

(**************************************************************)

type unmarsh = {
  buf : string ;
  mutable pos : int
} 

let unmarsh s = {
  buf = s ;
  pos = 0
} 

let get m len =
  if m.pos + len > String.length m.buf then
    failwith "sanity" ;
  let ret = String.sub m.buf m.pos len in
  m.pos <- m.pos + len ;
  ret

let read_int m =
  let s = get m 4 in
  Hsys.pop_int s 0

let read_bool m =
  let i = read_int m in
  match i with
  | 0 -> false
  | 1 -> true
  | _ -> failwith "read_bool:not a bool"

let read_list m f =
  let len = read_int m in
  let rec loop len =
    if len = 0 then [] 
    else (f ()) :: (loop (pred len))
  in loop len

let read_option m f =
  if read_bool m then 
    Some (f ())
  else None

let read_string m =
  let len = read_int m in
  let s = get m len in
  begin
    match (String.length s) mod 4 with
    | 0 -> ()
    | 1 -> get m 3 ; ()
    | 2 -> get m 2 ; ()
    | 3 -> get m 1 ; ()
    | _ -> failwith "sanity"
  end ;
  s

(**************************************************************)
