(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* IQ.ML *)
(* Author: Mark Hayden, 7/95 *)
(**************************************************************)
open Util
open Printf
(**************************************************************)
let name = "IQ"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

let debug = false

(* This is refered to in cbypass.c *)
type 'a t = {
  unset		 : 'a ;
  reset		 : 'a ;

  mutable head   : int ;
  mutable tail 	 : int ;
  mutable read	 : int ;

  mutable offset : int ;
  mutable max	 : int ;
  mutable array  : 'a array
}

(**************************************************************)

let fix_max iq =
  iq.max <- Array.length iq.array - iq.offset

(**************************************************************)

let align iq =
(*printf "IQ:realigning\n" ;*)
  let size = Array.length iq.array
  and array = iq.array
  and reset = iq.reset in

  let first =
    let rec find_first i =
      if i <| size && array.(i) == reset then
	find_first (succ i)
      else 
      	i
    in find_first 0
  in
  if first > 0 then (
    Array.blit iq.array first iq.array 0 (size-first) ;
    Array.fill iq.array (size-first) first iq.unset ;
    iq.offset <- iq.offset - first ;
    fix_max iq
  )

(**************************************************************)

let resize iq size =
(*
  if !Util.verbose then
    printf "IQ:resize\n" ;
*)
  align iq ;
  let size = size + iq.offset in
  if size * 2 + 10 >= Array.length iq.array then (
    let len = size*4+10 in
    let na = array_create name len iq.unset in
    for i = 0 to pred (Array.length iq.array) do
      na.(i) <- iq.array.(i)
    done ;
    iq.array <- na ;
    fix_max iq
  )

(**************************************************************)

let grow iq size =
  align iq ;
  let osize = Array.length iq.array in
  let size = max size osize in
  let na = array_create name size iq.unset in
  Array.blit iq.array 0 na 0 osize ;
  iq.array <- na ;
  fix_max iq

(**************************************************************)

let nth_raw iq i =
  let i = i + iq.offset in
  let array = iq.array in
  if i < 0 then
    iq.reset
  else if i >= Array.length array then
    iq.unset
  else array.(i)

let nth iq i =
  let it = nth_raw iq i in
  if it == iq.reset || it == iq.unset then
    None
  else 
    Some it

let nth_set iq i v =
  if i >= iq.max then
    resize iq (succ i) ;
  let i = i + iq.offset in
  let array = iq.array in
  if i >= 0 && array.(i) == iq.unset then (
    array.(i) <- v ; true
  ) else (
    false
  )

let nth_reset iq i =
  if i >= iq.max then
    resize iq (succ i) ;
  let i = i + iq.offset in
  if i >= 0 then
    iq.array.(i) <- iq.reset

(**************************************************************)

let get = nth_raw

(**************************************************************)

let empty unset reset = 
  (* These checks allow us to use '==' below instead of '='.
   *)
  if Obj.is_block (Obj.repr unset) then
    failwith "empty:unset is a block" ;
  if Obj.is_block (Obj.repr reset) then
    failwith "empty:reset is a block" ;

  let iq = {
    array 	= array_create name 4 unset ;
    unset 	= unset ;
    reset 	= reset ;
    head 	= 0 ;
    tail 	= 0 ;
    read  	= 0 ;
    offset 	= 0 ;
    max		= 0
  } in
  fix_max iq ;
  iq

and head iq = iq.head
and tail iq = iq.tail
and read iq = iq.read

and advance_tail iq n =
  if n > iq.tail then
    iq.tail <- n

and advance_head_gc iq n f =
  if n > iq.head then (
    for i = iq.head to pred n do
      match nth iq i with
      |	Some it -> 
	  f it ;
	  nth_reset iq i
      |	None -> ()
    done ;
    iq.head <- n ;
    if n > iq.tail then
      iq.tail <- n
  )

let advance_head iq n =
  if n > iq.head then (
    for i = iq.head to pred n do
      nth_reset iq i
    done ;
    iq.head <- n ;
    if n > iq.tail then
      iq.tail <- n
  )

let check iq =
  if iq.head > iq.tail then 
    failwith "head > tail" ;
  for i = 0 to pred (Array.length iq.array) do
    let j = i - iq.offset in
    let v = iq.array.(i) in
    if v <> iq.reset && j < iq.head then
      failwith "before head, not = reset" ;
    if v <> iq.unset && j > iq.tail then
      failwith "after tail, not = unset" ;
  done

(**************************************************************)

let assign iq i v =
  let ret = nth_set iq i v in
  if succ i > iq.tail then
    iq.tail <- succ i ;
  ret

let add iq v =
  let tail = iq.tail in
  iq.tail <- succ tail ;
  assign iq tail v ; ()

(**************************************************************)

let get_prefix iq f =
  let rec loop i =
    match nth iq i with
    | Some msg -> (
        f i msg ;
	loop (succ i)
      )
    | None -> advance_head iq i
  in loop iq.head

(**************************************************************)

let opt_check_update iq i =
  if iq.head = i 
  && iq.head = iq.tail
  then (
    nth_reset iq i ;
    let i = succ i in
    iq.head <- i ;
    iq.tail <- i ;
    iq.read <- i ;
    true
  ) else (
    false
  )

(**************************************************************)

let free iq f =
  let a = iq.array in
  for i = 0 to pred (Array.length a) do
    let it = a.(i) in
    if it <> iq.unset && it <> iq.reset then (
      f it ;
      a.(i) <- iq.reset
    )
  done

(**************************************************************)

let opt_insert_check_doread iq i v =
  if i = iq.read 
  && i = iq.tail
  then (
    if not (nth_set iq i v) then 
      failwith "sanity[1]" ;
    iq.tail <- succ iq.tail ;
    iq.read <- succ iq.read ;
    true
  ) else false

(**************************************************************)

let read_prefix iq f =
  let rec loop i =
    match nth iq i with
    | Some msg -> (
        f msg i ;
	loop (succ i)
      )
    | None -> iq.read <- i
  in loop iq.read

(**************************************************************)

let list_of_iq_interval iq (head,tail) =
  let rec loop i =
    if i > tail then []
    else match nth iq i with
    | Some v -> (i,v)::(loop (succ i))
    | None -> loop (succ i)
  in loop head

(**************************************************************)

let list_of_iq ({head=head; tail=tail} as iq) =
  list_of_iq_interval iq (head,tail)

(*
let list_of_iq ({head=head; tail=tail} as iq) =
  let l = ref [] in
  for i = 0 to pred (Array.length iq.array) do
    let v = iq.array.(i) in
    if v <> iq.unset && v <> iq.reset then (
      let j = i - iq.offset in
      l := (j,v) :: !l
    )
  done ;
  !l
*)

(**************************************************************)
(* HOLE: Returns the index of the start and end of any hole at
 * the beginning of the "window."  Used for NAKs...
 *)

let hole ({head=head; tail=tail} as iq) =
  if tail <= head then (
    None
  ) else (
    let first_set =
      let rec loop i =
	if i >= tail then i
	else (
	  match nth iq i with
	  | Some _ -> i
	  | None -> loop (succ i)
	)
      in loop head
    in
    if first_set <= head then None
    else Some (head, pred first_set)
  )

(**************************************************************)
(* READ_HOLE: Returns the index of the start and end of any
 * hole at the beginning of the "window."  Used for NAKs...
 *)

let read_hole iq =
  if iq.tail <= iq.read then (
    None
  ) else (
    let first_set =
      let rec loop i =
	if i >= iq.tail then i
	else (
	  match nth iq i with
	  | Some _ -> i
	  | None -> loop (succ i)
	)
      in loop iq.read
    in Some (iq.read, pred first_set)
  )

(**************************************************************)

let print iq =
  for i = 0 to pred (Array.length iq.array) do
    let j = i - iq.offset in
    let v = iq.array.(i) in
    let kind =
      if v == iq.reset then "reset"
      else if v == iq.unset then "unset"
      else "data"
    in
    let pointer = "" in
    let pointer = if j = iq.head then pointer ^ "<-head" else pointer in
    let pointer = if j = iq.read then pointer ^ "<-read" else pointer in
    let pointer = if j = iq.tail then pointer ^ "<-tail" else pointer in
    printf "  %d:%s:%s\n" j kind pointer ;
  done

(**************************************************************)
(* CHECK: checks some properties that should hold on Iq's.
 *)

let check iq =
  let failwith s = print iq ; failwith s in
  if iq.head > iq.tail then 
    failwith "head > tail" ;
  for i = 0 to pred (Array.length iq.array) do
    let j = i - iq.offset in
    let v = iq.array.(i) in
    if v <> iq.reset && j < iq.head then
      failwith "before head, not = reset" ;
    if v <> iq.unset && j > iq.tail then
      failwith "after tail, not = unset" ;
  done

(**************************************************************)
