(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* IOVECL.ML: operations on arrays of Iovec's. *)
(* Author: Mark Hayden, 5/96 *)
(**************************************************************)
open Trans
open Util
(**************************************************************)
let name = Trace.source_file "IOVECL"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type t = Iovec.t array

(**************************************************************)

let free debug il =
  for i = 0 to pred (Array.length il) do Iovec.free debug il.(i) done

let ref debug il =
  for i = 0 to pred (Array.length il) do Iovec.ref debug il.(i) done

let check debug il =
  for i = 0 to pred (Array.length il) do Iovec.check debug il.(i) done

(**************************************************************)

let len debug il = 
  let l = Pervasives.ref 0 in
  for i = 0 to pred (Array.length il) do
    l := !l + Iovec.len debug il.(i)
  done ;
  !l

(**************************************************************)

let flatten_buf debug il buf ofs max_len = 
  let len = len debug il in
  if len > max_len then
    failwith "flatten_buf:iovecl too large" ;

  Iovec.log (fun () -> sprintf "flatten:%s:%d -> %d bytes" debug (Array.length il) len) ;
  let ofs = Pervasives.ref ofs in
  for i = 0 to pred (Array.length il) do
    Iovec.read debug il.(i) (fun sbuf sofs slen ->
      String.blit sbuf sofs buf !ofs slen ;
      ofs := !ofs + slen
    )
  done ;
  len

(**************************************************************)

let flatten debug = 
(*
  let debug = "Iovec(flat):"^debug in
*)
  fun il ->
    match Array.length il with
    | 0 -> Iovec.empty debug
    | 1 -> 
	Iovec.ref debug il.(0) ;
	il.(0)
    | _ ->
	let len = len debug il in
	Iovec.log (fun () -> sprintf "flatten:%s:%d -> %d bytes" debug (Array.length il) len) ;
	let iov = Iovec.create debug len in
	Iovec.write_hack debug iov (flatten_buf debug il) ;
	iov

(**************************************************************)

let make_marsh debug =
  let marsh,unmarsh = Iovec.make_marsh debug in
  let marsh obj =
    let iov = marsh obj in
    [| iov |]
  and unmarsh iovl =
    let iov = flatten debug iovl in
    let obj = unmarsh iov in
    Iovec.free debug iov ;
    obj
  in (marsh,unmarsh)

(**************************************************************)

let clean debug il =
  let n = Array.length il in
  let nempty = Pervasives.ref 0 in
  for i = 0 to pred n do
    if Iovec.len debug il.(i) = 0 then
      incr nempty
  done ;
  if !nempty = 0 then (
    il
  ) else (
    let nl = array_create name (n - !nempty) (Iovec.empty debug) in
    nempty := 0 ;
    for i = 0 to pred n do
      if Iovec.len debug il.(i) = 0 then (
	Iovec.free debug il.(i) ;
      	incr nempty
      ) else (
	il.(i - !nempty) <- il.(i) ;
      )
    done ;
    nl
  )

(**************************************************************)

let fragment debug frag_len iovl =
  (* Round fragment size down to next lower
   * word
   *)
  let frag_len = floor_word frag_len in
  if frag_len <= 0 then failwith "sanity" ;

  let lenl = len debug iovl in
  let niov = Array.length iovl in
  let nfrags = (lenl + frag_len - 1) / frag_len in
  let fragi = array_create name (succ nfrags) 0 in
  let frago = array_create name (succ nfrags) 0 in

  (* Calculate boundaries of the fragments.
   *)
  let rec loop f fo i io =
(*
    printf "f=%d fo=%d i=%d io=%d\n" f fo i io ;
*)
    if f > nfrags then (
    ) else if i = niov then (
      fragi.(f) <- pred niov ;
      frago.(f) <- Iovec.len debug iovl.(pred niov) - frago.(f)
    ) else (* if f < nfrags && i < niov then*) (
      let il = Iovec.len debug iovl.(i) - io in
      let fl = frag_len - fo in
      if fl > il then (
      	loop f (fo + il) (succ i) 0
      ) else (
(*
      	printf "f.(%d) <- (%d,%d)\n" f i (io+fl) ;
*)
      	fragi.(f) <- i ;
      	frago.(f) <- (io+fl) ;
      	loop (succ f) 0 i (io + fl)
      )
    )
	
  in loop 1 0 0 0 ;

  (* Now break up the array of Iovec's into arrays
   * of fragments.
   *)
  let frags = array_create name nfrags [||] in
  for f = 0 to pred nfrags do
    let ni = succ (fragi.(succ f) - fragi.(f)) in
    let frag = Array.sub iovl fragi.(f) ni in
    frags.(f) <- frag ;

    match ni with
    | 0 -> failwith "sanity"
    | 1 ->
	Iovec.read debug frag.(0) (fun buf ofs len ->
	  let len = frago.(succ f) - frago.(f) in
	  frag.(0) <- Iovec.sub debug frag.(0) (ofs+frago.(f)) len
        )
    | _ ->
	Iovec.read debug frag.(0) (fun buf ofs len ->
	  frag.(0) <- Iovec.sub debug frag.(0) (ofs+frago.(f)) (len-frago.(f))
        ) ;

	(* Also refcount the intermediate iovecs.
	 *)
	for i = 1 to pred (pred ni) do
	  Iovec.ref debug frag.(i) ;
	done ;
	  
	let li = pred ni in
	Iovec.read debug frag.(li) (fun buf ofs len ->
	  frag.(li) <- Iovec.sub debug frag.(li) ofs (frago.(succ f))
        )
  done ;

  frags

(* Old code
let fragment =
  let iov = Iovecl.flatten name (getIov ev) in
  Iovec.get name iov (fun buf ofs len ->
    let n = (len + s.max_len - 1) / s.max_len in
    for i = 0 to pred n do
      let ofs = ofs + (i * s.max_len) in
      let len = min (len - i * s.max_len) s.max_len in
      let iov = [|Iovec.alloc name buf ofs len Iovec.heap|] in
    done
  ) 
*)
      
(**************************************************************)

let print name iovl = 
(*
  let s = flatten name iovl in
  let s = to_string name s in
  let len = String.length s in
  let s = hex_of_string s in
  log (fun () -> sprintf "%s:%d:%s\n" name len s)
*)()

(**************************************************************)
