(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* IOVEC.MLI *)
(* Author: Mark Hayden, 3/95 *)
(* Unless otherwise mentioned, these functions do not free
 * the reference counts of their arguments and iovecs
 * returned have all been correctly reference counted. *)
(**************************************************************)
open Trans
(**************************************************************)

(* Type of Iovecs.  They are opaque.
 *)
(*
type t
*)
type rbuf = string Refcnt.t
type t = {				(* this should be opaque *)
  rbuf : rbuf ;
  ofs : ofs ;
  len : len
}

(**************************************************************)
(* Constructors. *)

(* Create an empty iovec.
 *)
val empty 	: debug -> t

(* Create a new iovec as part of another iovec.  Offset is
 * the absolute offset in the buffer and is not relative to
 * the parent's offset.
 *)
val sub         : debug -> t -> ofs -> len -> t

(**************************************************************)
(* Reference count operations. *)

val free	: debug -> t -> unit
val ref		: debug -> t -> unit
val check	: debug -> t -> unit

(**************************************************************)
(* Accessors. *)

(* Open for reading.  The callback should not modify the
 * buffer.  
 *)
val read	: debug -> t -> (buf -> ofs -> len -> 'a) -> 'a

(* Similar to above, with different order of args.
 *)
val read_fun	: debug -> (t -> buf -> ofs -> len -> 'a) -> t -> 'a

(* Read first 4 bytes of iovec as a 32-bit integer in 
 * network-byte-order.
 *)
val read_int	: debug -> t -> int

(* Length of an iovec.
 *)
val len		: debug -> t -> len

(**************************************************************)

(* Marshallers for iovecs.  Allocation is done on the ML heap.
 *)
val make_marsh  : debug -> (('a -> t) * (t -> 'a))

(**************************************************************)
(* All of these do allocation on the ML heap.
 *)

val create	: debug -> len -> t
val of_string 	: debug -> string -> t
val of_substring : debug -> buf -> ofs -> len -> t

(**************************************************************)
(* For debugging purposes.
 *)
val to_string	: debug -> t -> string	(* allocates *)
val chop 	: debug -> t -> t
val shift 	: debug -> t -> t

(**************************************************************)

(**************************************************************)
val log : (unit -> string) -> unit	(* for Iovecl *)
(**************************************************************)
(**************************************************************)
(* These are not normally used directly.
 *)

(* Iovecs use reference-counted string buffers.
 *)
type rbuf = string Refcnt.t

(* The ML heap is a manager.
 *)
val heap : debug -> string -> rbuf	(* arg is the buffer *)

(* Allocate an iovec using a particular heap.  This
 * should not normally be used.  Allocation should
 * be done via the Mbuf module.
 *)
val alloc	: debug -> rbuf -> ofs -> len -> t

(* Open for writing.  This should not normally be used.
 *)
val write_hack	: debug -> t -> (buf -> ofs -> len -> 'a) -> 'a

(**************************************************************)
val break : debug -> t -> (rbuf -> ofs -> len -> 'a) -> 'a
