(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* IOVEC.ML *)
(* Author: Mark Hayden, 5/96 *)
(**************************************************************)
open Trans
open Util
(**************************************************************)
let name = Trace.source_file "IOVEC"
let failwith s = failwith (Util.failmsg name s)
let log = Trace.log name ""
(**************************************************************)

type manager = string Refcnt.t

let heap = Refcnt.void

type rbuf = string Refcnt.t

(**************************************************************)
(* Update ensemble/socket/skt.h, too 
 *)
type t = {
  rbuf : rbuf ;
  ofs : ofs ;
  len : len
}

let len debug iov = iov.len

(**************************************************************)

let free debug i  = Refcnt.decr debug i.rbuf
let ref debug i   = Refcnt.incr debug i.rbuf
let check debug i = Refcnt.check debug i.rbuf

(**************************************************************)

(* Wrap a string as an Iovec.  If it is not word aligned,
 * then first copy it to a new string.
 *)
let alloc debug rbuf ofs len = 
  let buf = Refcnt.read debug rbuf in

  if not (is_aligned ofs) || not (is_aligned len) then
    failwith "alloc:ofs or len not aligned" ;
  if ofs <| 0 || ofs + len > String.length buf then
    failwith "alloc:bad ofs or len" ;

  Refcnt.incr debug rbuf ;
  { rbuf = rbuf ;
    ofs = ofs ; 
    len = len
  }

(**************************************************************)

let read debug iov f = 
  f (Refcnt.read debug iov.rbuf) iov.ofs iov.len

let read_fun debug f iov =
  f iov  (Refcnt.read debug iov.rbuf) iov.ofs iov.len

let read_int debug iov = 
  Hsys.pop_int (Refcnt.read debug iov.rbuf) iov.ofs

let write_hack debug iov f = 
  f (Refcnt.read debug iov.rbuf) iov.ofs iov.len

(**************************************************************)

let sub debug iov ofs len = 
  alloc debug iov.rbuf ofs len

(**************************************************************)

let empty = 
  let s = "" in
  alloc "IOVEC:empty" (heap "empty" s) 0 0

let empty debug = 
  ref debug empty ;
  empty

(**************************************************************)

let create debug len =
  let buf = String.create len in
  let rbuf = heap debug buf in
  alloc debug rbuf 0 len

(**************************************************************)

let to_string debug iov =
  read debug iov (fun buf ofs len -> 
    log (fun () -> sprintf "to_string:%s:len=%d" debug len) ;
    String.sub buf ofs len
  )

(**************************************************************)

let of_substring debug buf ofs len =
  log (fun () -> sprintf "of_substring:%s:len=%d" debug len) ;
  let str = String.sub buf ofs len in
  alloc debug (heap debug str) 0 (String.length str)

(**************************************************************)

let of_string debug str =
  log (fun () -> sprintf "of_string:%s:len=%d" debug (String.length str)) ;
  let str = String.copy str in
  alloc debug (heap debug str) 0 (String.length str)

(**************************************************************)

let make_marsh debug =
  let (marsh,unmarsh) = Util.make_marsh debug true in
  let marsh obj =
    let str = marsh obj in
    alloc debug (heap debug str) 0 (String.length str)
  and unmarsh iov =
    read debug iov unmarsh
  in (marsh,unmarsh)

(**************************************************************)

let chop debug iov =
  read debug iov (fun buf ofs len ->
    let buf = String.sub buf ofs len in
    alloc debug (heap debug buf) 0 len
  )

let shift debug iov =
  read debug iov (fun buf ofs len ->
    let buf = "grbg"^buf in
    alloc "shift" (heap debug buf) (ofs+4) len
  )

(**************************************************************)
let break debug iov f = f iov.rbuf iov.ofs iov.len
