(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* HSYS.ML *)
(* Author: Mark Hayden, 5/95 *)
(**************************************************************)
let name = "HSYS"
let failwith s = failwith (failwith (name^":"^s))
(**************************************************************)
open Printf
(**************************************************************)
(* From util.ml *)
let word_len = 4
let mask1 = pred word_len
let mask2 = lnot mask1
let ceil_word i = (i + mask1) land mask2
(**************************************************************)
external (>|) : int -> int -> bool = "%gtint"
(**************************************************************)

type buf = string
type ofs = int
type len = int
type port = int
type socket = Unix.file_descr
type inet = Unix.inet_addr
type info = Socket.info

type iovec = buf * ofs * len

type handler = Socket.handler =
  | Handler0 of (unit -> unit)
  | Handler1

type socket_option =
  | Nonblock
  | Reuse
  | Join of inet
  | Leave of inet
  | Multicast of bool
  | Sendbuf of int
  | Recvbuf of int

(**************************************************************)

let unix_wrap debug f =
  try f () with Unix.Unix_error(err,s1,s2) as exc ->
    match err with 
    | Unix.ECONNREFUSED | Unix.ECONNRESET | Unix.EPIPE -> 0
    | _ ->
	eprintf "HSYS_UNIX:%s:%s\n" debug (Unix.error_message err) ;
	  flush stderr ;
	  raise exc
	  
(**************************************************************)

let close 		= Unix.close
let getlogin 		= Unix.getlogin
let gettimeofdaya 	= Socket.gettimeofday
let has_ip_multicast 	= Socket.has_ip_multicast
let inet_any () 	= Unix.inet_addr_any
let int_of_socket 	= Socket.int_of_file_descr
let listen 		= Unix.listen
let max_msg_len () 	= 9*1024	(* suggested by Werner Vogels *)
let pop_int 		= Socket.pop_int
let preprocess s a 	= Socket.preprocess s [] (Array.map (fun (i,p) -> Unix.ADDR_INET(i,p)) a)
let push_int 		= Socket.push_int
let read                = Socket.read
let recv s b o l 	= unix_wrap "recv" (fun () -> Socket.recv s b o l)
let select 		= Socket.select
let send s b o l 	= unix_wrap "send" (fun () -> Socket.send s b o l)
let sendopt 		= Socket.sendopt
let sendvopt 		= Socket.sendvopt
let socket_dgram () 	= Unix.socket Unix.PF_INET Unix.SOCK_DGRAM 0
let socket_stream () 	= Unix.socket Unix.PF_INET Unix.SOCK_STREAM 0
let stdin 		= Socket.stdin
let static_string       = Socket.static_string
let static_string_free  = Socket.static_string_free
let marshal             = Socket.marshal
let bind sock inet port = Unix.bind sock (Unix.ADDR_INET(inet,port))
let connect sock inet port = Unix.connect sock (Unix.ADDR_INET(inet,port))

(**************************************************************)

let gethostname () =
  let name = Unix.gethostname () in
  String.lowercase name

(**************************************************************)

let gettimeofday () =
  let res = [|0.0|] in
  Socket.gettimeofday res ;
  res.(0)

(**************************************************************)

let setsockopt sock = function
  | Nonblock    -> Unix.set_nonblock sock
  | Reuse       -> Unix.setsockopt sock Unix.SO_REUSEADDR true
  | Join inet   -> Socket.setsock_join sock inet
  | Leave inet  -> Socket.setsock_leave sock inet
  | Multicast loopback -> Socket.setsock_multicast sock loopback
  | Sendbuf len -> Socket.setsock_sendbuf sock len
  | Recvbuf len -> Socket.setsock_recvbuf sock len

(**************************************************************)

let recvfrom s b o l = 
  let l,sa = Unix.recvfrom s b o l [] in
  match sa with 
  | Unix.ADDR_INET(i,p) -> (l,i,p)
  | _ -> failwith "recv_from:non-ADDR_INET"

(**************************************************************)

let error e =
  match e with
  | Unix.Unix_error(err,s1,s2) ->
      let msg = try Unix.error_message err with
      | _ -> "unknown error"
      in
      let s1 = if s1 = "" then "" else ", "^s1 in
      let s2 = if s2 = "" then "" else ", "^s2 in
      sprintf "Unix(%s%s%s)" msg s1 s2
  | Invalid_argument s -> 
      raise e
  | Failure s -> 
      sprintf "Failure(%s)" s
  | Sys_error s ->
      sprintf "Sys_error(%s)" s
  | _ -> 
      failwith "unknown error"

(**************************************************************)

let inet_of_string inet =
  try 
    Unix.inet_addr_of_string inet
  with _ ->
    let he = Unix.gethostbyname inet in	
    he.Unix.h_addr_list.(0)

let accept sock =
  match Unix.accept sock with
  | (sock,Unix.ADDR_INET(inet,port)) -> (sock,inet,port)
  | _ -> failwith "accept:non-ADDR_INET"

let inet_of_name name =
  let hostentry = Unix.gethostbyname name in
  let host      = hostentry.Unix.h_addr_list.(0) in
  host

(**************************************************************)

let deering_prefix = "224.0.0."		(* same as in ipmc.ml *)

let deering_addr i =
  let i = i mod 248 in
  let inet_s = sprintf "%s%d" deering_prefix i in
  Unix.inet_addr_of_string inet_s

(**************************************************************)

let string_of_inet inet =
  try
    let {Unix.h_name=long_name} =
      Unix.gethostbyaddr inet 
    in
    let rec loop i =
      if i >= String.length long_name then i
      else if long_name.[i] = '.' then i
      else loop (i+1)
    in 
    let name = String.sub long_name 0 (loop 0) in
    let name = String.lowercase name in
    name
  with _ ->
    let inet_s = Unix.string_of_inet_addr inet in
    let ofs = String.length deering_prefix
    and len = String.length inet_s in
    if deering_prefix = String.sub inet_s 0 (min ofs len) then
      sprintf "IPM.%s" (String.sub inet_s ofs (len-ofs))
    else
      inet_s

(**************************************************************)

type manager = Socket.manager
let poll = Socket.poll
let manager = Socket.manager
let fork = Socket.fork

(**************************************************************)

let read_ch ch =
  let s = ref "" in
  begin 
    try while true do
      let buf = String.create 100 in
      let len = input ch buf 0 (String.length buf) in
      if len = 0 then 
      	raise End_of_file ;
      s := !s ^ (String.sub buf 0 len) 
    done with End_of_file -> () 
  end ;
  !s

let open_process cmd input =
  let (in_read, in_write) = Unix.pipe() in
  let (out_read, out_write) = Unix.pipe() in
  let (err_read, err_write) = Unix.pipe() in

  match Socket.fork() with
  | 0 ->
      Unix.dup2 out_read Unix.stdin ;
      Unix.dup2 in_write Unix.stdout ;
      Unix.dup2 err_write Unix.stderr ;
      List.iter Unix.close  [in_read; out_write; err_write; out_read; in_write; err_read] ;
      Unix.execv "/bin/sh" [| "/bin/sh"; "-c"; cmd |];
      exit 127
  | pid ->
      let inchan  = Unix.in_channel_of_descr in_read in
      let outchan = Unix.out_channel_of_descr out_write in
      let errchan = Unix.in_channel_of_descr err_read in
      Unix.close out_read;
      Unix.close in_write;
      Unix.close err_write;

      output_string outchan input ; 
      close_out outchan ;

      let output = read_ch inchan in
      close_in inchan ;
      let error = read_ch errchan in
      close_in errchan ;

      let stat = snd(Unix.waitpid [] pid) in
      let stat = stat = Unix.WEXITED(0) in
      (stat,output,error)

(**************************************************************)
