(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* HANDLER.ML: handler tables *)
(* Author: Mark Hayden, 12/95 *)
(**************************************************************)
let name = "HANDLER"
let failure s = failwith (name^":"^s)
(**************************************************************)
open Printf
(**************************************************************)

module type S = 
  sig
    type key
    type ('subkey,'data,'upcall) t

    val create 	: ('data array -> 'upcall) -> ('subkey,'data,'upcall) t
    val add 	: ('subkey,'data,'upcall) t -> key -> 'subkey -> 'data -> unit
    val remove 	: ('subkey,'data,'upcall) t -> key -> 'subkey -> unit
    val find 	: ('subkey,'data,'upcall) t -> key -> 'upcall
    val list_of : ('subkey,'data,'upcall) t -> (key * 'subkey) list
    val size 	: ('subkey,'data,'upcall) t -> int
    val info	: ('subkey,'data,'upcall) t -> string
  end

module Make(H: Hashtbl.S) : (S with type key = H.key) =
  struct
    type key = H.key
    type ('subkey,'data,'upcall) t = 
      { merge : 'data array -> 'upcall ;
	table : ('upcall * (('subkey * 'data) array)) H.t ;
        empty : 'upcall }

    let create merge = 
      { table = H.create 100 ;
      	merge = merge ;
        empty = merge [||] } 

    let list_of t =
      let l = ref [] in
      H.iter (fun key (upcall,entries) ->
	Array.iter (fun (subkey,item) ->
	  l := (key,subkey) :: !l
	) entries
      ) t.table ;
      !l

    let add t key subkey data =
      let entries =
	try 
	  let entries = snd (H.find t.table key) in
	  H.remove t.table key ;
	  entries
	with Not_found -> [||]
      in
      Array.iter (fun (sk,_) ->
	if subkey = sk then
	  failwith "multiple handlers for same subkey"
      ) entries ;
      let entries = Array.append entries [|(subkey,data)|] in
      let merge = t.merge (Array.map snd entries) in
      H.add t.table key (merge,entries)
    (*;check t*)

    let remove t key subkey =
      let entries =
      	try snd (H.find t.table key)
	with Not_found -> failwith "remove:not found"
      in
      H.remove t.table key ;
      let entries = Array.to_list entries in
      let found = ref false in
      let entries =
	List.fold_left (fun l (sk,d) ->
	  if subkey = sk then (
	    if !found then
	      failwith "remove:found more than one" ;
	    found := true ;
	    l
	  ) else (sk,d) :: l
	) [] entries
      in
      if not !found then
	failwith "remove:did not find item" ;
      if entries <> [] then (
	let entries = Array.of_list entries in
	let merge = t.merge (Array.map snd entries) in
	H.add t.table key (merge,entries)
      (* ; check t*)
      )

    let find t = 
      let ht = t.table in
      fun key ->
      	try 
	  (fst (H.find t.table key))
      	with Not_found -> 
	  t.empty

    let size t =
      let counter = ref 0 in
      H.iter (fun _ _ -> incr counter) t.table ;
      !counter

    let total_size t =
      let counter = ref 0 in
      H.iter (fun _ (_,b) -> 
	counter := !counter + Array.length b
      ) t.table ;
      !counter

    let info t =
      sprintf "{Handler:#ids=%d:total=%d}" (size t) (total_size t)
  end

(**************************************************************)
