(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* VIEW.MLI *)
(* Author: Mark Hayden, 3/96 *)
(**************************************************************)
open Trans
(**************************************************************)

(* VIEW.T: an array of endpt id's.  These arrays should
 * be treated as though they were immutable.
 *)
type t = Endpt.id array

val to_string : t -> string

(**************************************************************)

(* VIEW.ID: a logical time and an endpoint id.  View
 * identifiers have a lexicographical total ordering.  
 *)
type id = ltime * Endpt.id

(* Display function for view id's.
 *)
val string_of_id : id -> string

(**************************************************************)

(* VIEW.STATE: a record of information kept about views.
 * This value should be common to all members in a view.
 *)
type state = {
  version       : Version.id ;		(* version of Ensemble *)
  coord         : rank ;		(* initial coordinator *)
  group		: Group.id ;		(* name of group *)
  view 		: t ;			(* members in the view *)
  address       : Addr.set array ;	(* addresses of members *)
  view_id 	: id ;			(* unique id of this view *)
  params        : Param.tl ;		(* parameters of protocols *)
  prev_ids      : id list ;             (* identifiers for prev. views *)
  groupd        : bool ;		(* using groupd server? *)
  proto_id	: Proto.id ;		(* id of protocol in use *)
  xfer_view	: bool ;		(* is this an XFER view? *)
  key		: Security.key ;	(* keys in use *)
  clients	: bool array ;		(* who are the clients in the group? *)
  primary       : bool ;		(* primary partition? (only w/some protocols) *)
  transitional  : bool			(* transitional view (for Transis) *)
}

(* FIELDS: these correspond to the fields of a View.state.
 *)
type fields =
  | Vs_coord        of rank
  | Vs_group	    of Group.id
  | Vs_view 	    of t
  | Vs_view_id 	    of id
  | Vs_params	    of Param.tl
  | Vs_prev_ids     of id list
  | Vs_proto_id	    of Proto.id
  | Vs_xfer_view    of bool
  | Vs_key	    of Security.key
  | Vs_clients	    of bool array
  | Vs_groupd       of bool
  | Vs_transitional of bool
  | Vs_primary      of bool
  | Vs_address      of Addr.set array

(* SET: Construct new view state based on previous view state.
 *)
val set : state -> fields list -> state

val string_of_state : state -> string

(**************************************************************)

(* VIEW.LOCAL: information about a view that is particular to 
 * a member.
 *)
type local = {
  endpt	        : Endpt.id ;		(* endpoint id *)
  addr	        : Addr.set ;		(* my address *)
  rank 	        : rank ;		(* rank in the view *)  
  name		: string ;		(* my string name *)
  nmembers 	: nmembers ;		(* # members in view *)
  am_coord      : bool  		(* rank = vs.coord? *)
}  

(* LOCAL: create local record based view state and endpt.
 *)
val local : debug -> Endpt.id -> state -> local

(**************************************************************)

(* VIEW.FULL: a full view state contains both local and common
 * information.
 *)
type full = local * state

(* SINGLETON: Create full view state for singleton view.
 *)
val singleton :
  Security.key ->			(* security key *)
  Proto.id ->				(* protocol *)
  Group.id ->				(* group *)
  Endpt.id ->				(* endpt *)
  Addr.set ->				(* addresses *)
  full

(* STRING_OF_FULL: Display function for view state records.
 *)
val string_of_full : full -> string

(* CHECK: check some conditions on the sanity of the 
 * view state.
 *)
val check : debug -> full -> unit

(**************************************************************)

val addr : full -> Addr.set
val endpt_full : full -> Endpt.full 

(**************************************************************)
