(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* VIEW.ML *)
(* Author: Mark Hayden, 3/96 *)
(**************************************************************)
open Util
open Trans
(**************************************************************)

type t = Endpt.id array

let to_string = string_of_array Endpt.string_of_id

(**************************************************************)

type id = ltime * Endpt.id

(**************************************************************)

type state = {
  version       : Version.id ;		(* version of Ensemble *)
  coord         : rank ;		(* initial coordinator *)
  group		: Group.id ;		(* name of group *)
  view 		: t ;			(* list of members in the view *)
  address       : Addr.set array ;	(* addresses of members *)
  view_id 	: id ;			(* unique id of this view *)
  params        : Param.tl ;		(* parameters of protocols *)
  prev_ids      : id list ;             (* identifiers for prev. views *)
  groupd        : bool ;		(* using groupd server? *)
  proto_id	: Proto.id ;		(* id of protocol in use *)
  xfer_view	: bool ;		(* is this an XFER view? *)
  key		: Security.key ;	(* keys in use *)
  clients	: bool array ;		(* who are the clients in the group? *)
  primary       : bool ;		(* primary partition? (only w/some protocols) *)
  transitional  : bool			(* transitional view (for Transis) *)
}

(**************************************************************)

type local = {
  endpt	        : Endpt.id ;		(* endpoint id *)
  addr	        : Addr.set ;		(* my address *)
  rank 	        : rank ;		(* rank in the view *)
  name		: string ;		(* my string name *)
  nmembers 	: nmembers ;		(* # members in view *)
  am_coord      : bool  		(* rank = vs.coord? *)
}  

(**************************************************************)

type full = local * state

(**************************************************************)

let name rank endpt =
  sprintf "(%d,%s)" rank (Endpt.string_of_id endpt)

let check dbg (ls,vs) =
  let failwith s =
    failwith (sprintf "check:%s:%s" dbg s)
  in
  if ls.name <> name ls.rank ls.endpt then
    failwith "bad name" ;
  if not (array_mem ls.endpt vs.view) then
    failwith "endpt not in view" ;
  if ls.rank <> array_index ls.endpt vs.view then
    failwith "bad rank" ;
  if ls.nmembers <> Array.length vs.view then
    failwith "bad nmembers<>length view" ;
  if ls.nmembers <> Array.length vs.address then
    failwith "bad nmembers<>length address"

(**************************************************************)

let local debug endpt vs =
  let rank = array_index endpt vs.view in
  let name = name rank endpt in
  let nmembers = Array.length vs.view in
  { name = name ;
    endpt = endpt ;
    addr = vs.address.(rank) ;
    nmembers = nmembers ;
    rank = rank ;
    am_coord = rank = vs.coord
  } 

let singleton key proto_id group endpt addr =
  let vs = {
    coord       = 0 ;
    group	= group ;
    version     = Version.id ;
    view   	= [|endpt|] ;
    address     = [|addr|] ;
    view_id  	= (0,endpt) ;
    params      = [] ;
    prev_ids    = [] ;
    proto_id 	= proto_id ;
    key		= key ;
    xfer_view 	= false ;
    groupd      = false ;
    clients	= [||] ;
    transitional = false ;
    primary     = false
  } in

  let ls = local name endpt vs in
  (ls,vs)

(**************************************************************)

type fields =
  | Vs_coord        of rank
  | Vs_group	of Group.id
  | Vs_view 	of t
  | Vs_view_id 	of id
  | Vs_params	of Param.tl
  | Vs_prev_ids of id list
  | Vs_proto_id	of Proto.id
  | Vs_xfer_view of bool
  | Vs_key	of Security.key
  | Vs_clients	of bool array
  | Vs_groupd   of bool
  | Vs_transitional of bool
  | Vs_primary  of bool
  | Vs_address  of Addr.set array

(**************************************************************)

let string_of_id (ltime,ept) = 
  sprintf "(%d,%s)" ltime (Endpt.string_of_id ept)

let string_of_full (ls,vs) =
  sprintf "View.full{endpt=%s;group=%s;coord=%d;view=%s;rank=%d;nmembers=%d;view_id=%s;prev_ids=%s;proto_id=%s;xfer_view=%b;key=%s;clients=%s;primary=%b}"
    (Endpt.string_of_id ls.endpt)
    (Group.string_of_id vs.group)
    vs.coord
    (to_string vs.view)
    ls.rank
    ls.nmembers
    (string_of_id vs.view_id)
    (string_of_list string_of_id vs.prev_ids)
    (Proto.string_of_id vs.proto_id)
    vs.xfer_view
    (Security.string_of_key vs.key)
    (string_of_bool_array vs.clients)
    vs.primary

let string_of_state vs =
  sprintf "View.state{group=%s;coord=%d;view=%s;view_id=%s;prev_ids=%s;proto_id=%s;xfer_view=%b;key=%s;clients=%s;primary=%b}"
    (Group.string_of_id vs.group)
    vs.coord
    (to_string vs.view)
    (string_of_id vs.view_id)
    (string_of_list string_of_id vs.prev_ids)
    (Proto.string_of_id vs.proto_id)
    vs.xfer_view
    (Security.string_of_key vs.key)
    (string_of_bool_array vs.clients)
    vs.primary

let rec set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo vsl = match vsl with
  | [] -> {version=ver;group=gro;view=vie;view_id=vid;proto_id=pro;xfer_view=xfe;key=key;clients=cli;prev_ids=pre;params=par;groupd=gpd;primary=pri;transitional=tra;address=add;coord=coo}
  | hd :: tl -> match hd with
    | Vs_coord	   coo -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_group	   gro -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_view 	   vie -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_address   add -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_view_id   vid -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_params    par -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_prev_ids  pre -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_proto_id  pro -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_xfer_view xfe -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_key	   key -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_clients   cli -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_groupd    gpd -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_transitional tra -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
    | Vs_primary   pri -> set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo tl
	
let set {version=ver;group=gro;view=vie;view_id=vid;proto_id=pro;xfer_view=xfe;key=key;clients=cli;prev_ids=pre;params=par;groupd=gpd;primary=pri;transitional=tra;address=add;coord=coo} vsl =
  set_loop ver gro vie vid pro xfe key cli pre par gpd pri tra add coo vsl

(**************************************************************)

let addr (ls,vs) = vs.address.(ls.rank)

let endpt_full (ls,vs) = (ls.endpt, (addr (ls,vs)))

(**************************************************************)
