(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* UNIQUE.ML *)
(* Author: Mark Hayden, 8/96 *)
(**************************************************************)
let name = Trace.source_file "UNIQUE"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)
open Util
open Trans
open Hsys
(**************************************************************)

(* An id consists of an IP address, UDP port number, randomized
 * incarnation number, and a multiplexing number (mux).
 *)
type id	= {
  mux 		: mux ;
  host 		: inet ;
  port 		: port ;
  incarn 	: incarn
}

(* The host is the IP address of this machine.  This should be
 * unique across processes on the system when used in
 * combination with a UDP port number that this process has
 * bound to.  
 *)
let host = 
  just_once (fun () ->
    try
      Hsys.inet_of_string (gethostname ())
    with e -> (
      eprintf "UNIQUE:warning:error looking up local IP address\n" ;
      eprintf "  (I'm assuming you are running simulations and will use null IP address)\n" ;
      Hsys.inet_of_string "0.0.0.0"
    )
  )

let port = ref None

(* Install_port is called elsewhere in the system to provide
 * the unique port number.  
 *)
let install_port p = 
  match !port with
  | Some _ -> ()			(* Do nothing, already called *)
  | None ->
      port := Some p

(* This is a counter that is incremented with each id that
 * is created.  This makes id's within a process unique.  
 *)
let counter = counter ()

(* This is an additional 3-digit number that is chosen randomly
 * based on the system clock.  It is thrown in just for the hell
 * of it.
 *)
let incarn = 
  just_once (fun () ->
    let incarn = gettimeofday () in
    let incarn = sprintf "%f" incarn in
    let incarn = Digest.string incarn in
    let incarn = Hashtbl.hash incarn in
    abs (incarn mod 1000)
  )

let warning = just_once (fun () ->
    eprintf "UNIQUE:warning:Unique.id called prior to Unique.install_port\n" ;
    eprintf "  (this may prevent some Ensemble identifiers from being unique)\n"
  )

let id () =
  let port = 
    match !port with
    | None -> 
      	warning () ;
	(-1)
    | Some port -> port
  in {
    host	= host () ;
    port 	= port ;
    incarn 	= incarn () ;
    mux 	= counter ()
  }

let mux id = id.mux

let hash_of_id = Hashtbl.hash

let string_of_id_realshort id = 
  string_of_int id.mux

let string_of_id_short id =
  let port = 
    if id.port = -1 then "" else sprintf ":%d" id.port
  in

  sprintf "%s%s:%d:%d" 
    (string_of_inet id.host) 
    port 
    (id.incarn mod 1000)
    id.mux

let string_of_id id =
  sprintf "{%s}" (string_of_id_short id)

let same_process a b =
     a.host = b.host
  && a.port = b.port
  && a.incarn = b.incarn

(**************************************************************)
