(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* TYPEDESCR.ML *)
(* Author: Robbert vanRenesse, 4/97 *)
(**************************************************************)
open Util

type type_constr =
  | CONST
  | NCCONST of t list
and t =
  | ABSTRACT
  | UNIT
  | BOOL of string
  | CHAR of string
  | INT of string
  | FLOAT of string
  | STRING of string
  | LIST of t
  | ARRAY of t
  | PRODUCT of t list
  | STRUCT of (string * t) list
  | CONSTR of (string * type_constr) list
  | FUNCTION of t list

let rec concat pf sep = function
  | [] ->
      ""
  | [hd] ->
      pf hd
  | hd :: tl ->
      (pf hd) ^ sep ^ (concat pf sep tl)

let rec sprint_type = function
  | ABSTRACT -> ""
  | UNIT -> "unit"
  | BOOL _ -> "bool"
  | CHAR _ -> "char"
  | INT _ -> "int"
  | FLOAT _ -> "float"
  | STRING _ -> "string"
  | LIST t ->
      "(" ^ (sprint_type t) ^ ") list"
  | ARRAY t ->
      "(" ^ (sprint_type t) ^ ") array"
  | PRODUCT l ->
      let print x = sprintf "(%s)" (sprint_type x) in
      concat print " * " l
  | STRUCT l ->
      let print (n, t) =
	(sprintf "%s : " n) ^ (sprint_type t)
      in
      "{ " ^ (concat print "; " l) ^ " }"
  | CONSTR l ->
      let print (n, d) = match d with
        | CONST ->
	    n
	| NCCONST tl ->
	    (sprintf "%s of (" n) ^
	    (concat sprint_type ") * (" tl) ^ ")"
      in
      concat print " | " l
  | FUNCTION l ->
      let print x = sprintf "(%s)" (sprint_type x) in
      concat print " -> " l

let rec pretty_sprint t v = match t with
  | ABSTRACT -> "<abstract>"
  | UNIT -> "()"
  | BOOL _ -> if (Obj.magic v) then "true" else "false"
  | CHAR _ -> sprintf "%c" (Obj.magic v)
  | INT _ -> sprintf "%d" (Obj.magic v)
  | FLOAT _ -> sprintf "%f" (Obj.magic v)
  | STRING "p_string" -> sprintf "\"%s\"" (Obj.magic v)
  | LIST t ->
      "[" ^ (concat (pretty_sprint t) "; " (Obj.magic v)) ^ "]"
  | ARRAY t ->
      "[|" ^ (concat (pretty_sprint t) "; " (Array.to_list (Obj.magic v))) ^ "|]"
  | PRODUCT l ->
      let values = Array.to_list (Obj.magic v) in
      let l = List.combine l values in
      let print (t, v) = pretty_sprint t v in
      "(" ^ (concat print ", " l) ^ ")"
  | STRUCT l ->
      let values = Array.to_list (Obj.magic v) in
      let l = List.combine l values in
      let print ((n, t), v) =
	sprintf "%s = %s" n (pretty_sprint t v)
      in
      "{" ^ (concat print "; " l) ^ "}"
  | CONSTR l ->
      let rec scan_const c = function
        | (n, CONST) :: tl ->
	    if c = 0 then
	      sprintf "%s" n
	    else
	      scan_const (c - 1) tl
	| (n, _) :: tl ->
	    scan_const c tl
	| [] ->
	    raise Not_found
      and scan_ncconst c = function
        | (n, CONST) :: tl ->
	    scan_ncconst c tl
	| (n, NCCONST l) :: tl ->
	    if c = 0 then
	      let values = Array.to_list (Obj.magic v) in
	      let l = List.combine l values in
	      let print (t, v) = pretty_sprint t v in
	      sprintf "%s(%s)" n (concat print ", " l)
	    else
	      scan_ncconst (c - 1) tl
	| [] ->
	    raise Not_found
      in
      if Obj.is_block (Obj.repr v) then
	scan_ncconst (Obj.tag (Obj.repr v)) l
      else
	scan_const (Obj.magic v) l
  | _ as t ->
      sprintf "<%s>" (sprint_type t)

let abstract_t = ABSTRACT
let unit_t = UNIT
let bool_t = BOOL "bool"
let char_t = CHAR "char"
let int_t = INT "int"
let float_t = FLOAT "float"
let string_t = STRING "string"
let list_t a = LIST a
let array_t a = ARRAY a
