(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* ROUTE.ML *)
(* Author: Mark Hayden, 12/95 *)
(**************************************************************)
open Trans
open Util
(**************************************************************)
let name = Trace.source_file "ROUTE"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

(* Type of routers.
 *)
type 'msg t = {
  name		: string ;
  secure	: bool ;
  blast 	: 
    (buf -> ofs -> len -> unit) ->	(* Xmit *)
    (Iovecl.t -> unit) ->		(* Xmit vect *)
    Security.key -> Conn.id -> 'msg ;
  install	: Conn.t -> Security.key -> (Conn.kind -> rank -> 'msg) -> unit ;
  remove	: Conn.t -> unit
}

type message =
  | Signed of (Conn.kind * rank * (Conn.kind -> rank -> bool -> int -> Obj.t option -> Iovecl.t -> unit))
  | Unsigned of (int -> Obj.t option -> Iovecl.t -> unit)
  | Bypass of (int -> Iovec.t -> unit)
  | Raw of (Iovecl.t -> unit)
  | Scale of (rank -> int -> Obj.t option -> Iovecl.t -> unit)

(**************************************************************)

let empty3 _ _ _ = ()(* Need printout *)

let merge1f a f = f (fun a1 -> Array.iter (fun f -> f a1) a)
let merge2f a f = f (fun a1 a2 -> Array.iter (fun f -> f a1 a2) a)
let merge3f a f = f (fun a1 a2 a3 -> Array.iter (fun f -> f a1 a2 a3) a)

let merge1 a = fun a1 -> Array.iter (fun f -> f a1) a
let merge2 a = fun a1 a2 -> Array.iter (fun f -> f a1 a2) a

let merge3 a = 
  match (Array.to_list a) with
  | [] -> empty3
  | [f1] -> f1
  | [f1;f2] -> fun a1 a2 a3 -> f1 a1 a2 a3 ; f2 a1 a2 a3
  | _ ->
      let n = pred (Array.length a) in
      fun a1 a2 a3 ->
	for i = 0 to n do
	  a.(i) a1 a2 a3
	done

let merge4 a = fun a1 a2 a3 a4 -> Array.iter (fun f -> f a1 a2 a3 a4) a

(**************************************************************)

module Int_key =
  struct
    type t = int
    let equal = (=|)
    let hash i = abs (i lsr 1)
  end

module Handler = Handler.Make ( Hashtbl.Make ( Int_key ) )

type merge = (Conn.id * Security.key * Conn.kind * rank * message) array -> (Iovec.t -> unit)

let merge info =
  let info = Array.map (fun (c,p,s,u,m) -> m [|(c,p,s,u)|]) info in
  merge3 info

let handlers = Handler.create merge
let deliver = Handler.find handlers
let drop = Trace.log "ROUTED" ""
let info = Trace.log "ROUTEI" ""

(*
let deliver i iov =
  let iv = Iovec.read name iov (fun buf ofs len ->
    Hsys.pop_int buf ofs
  ) in
  printf "ROUTE:deliver:pre-int=%x, int=%x, equal=%b\n" i iv (i = iv) ;
  deliver i iov
*)

let make name proc_conns proc_hdlr pack_of_conn merge blast =
  let install conns key hdlr =
    let id = conns.Conn.id in
    let conns = proc_conns conns.Conn.all_recv in
    List.iter (fun (conn,kind,rank,dbg_name) ->
      let pack = pack_of_conn conn in
      let pack0 = Hsys.pop_int pack 0 in
      let hdlr = proc_hdlr hdlr kind rank in
      Handler.add handlers pack0 id (conn,pack,key,hdlr,merge)
    ) conns
  in

  let remove conns =
    let id = conns.Conn.id in
    let conns = proc_conns conns.Conn.all_recv in
    List.iter (fun (conn,_,_,_) ->
      let pack = pack_of_conn conn in
      let pack0 = Hsys.pop_int pack 0 in
      Handler.remove handlers pack0 id
    ) conns
  in

  let blast xmit xmitv key conn =
    info (fun () -> sprintf "conn=%s" (Conn.string_of_id conn)) ;
    let pack = pack_of_conn conn in
    blast xmit xmitv key pack conn
  in

  { name	= name ;
    secure	= false ; (*BUG*)
    blast	= blast ;
    install	= install ;
    remove	= remove }

(**************************************************************)
