(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* PROPERTY.ML *)
(* Author: Mark Hayden, 12/96 *)
(**************************************************************)
open Util
(**************************************************************)
let name = Trace.source_file "PROPERTY"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type id =
  | Agree				(* agreed (safe) delivery *)
  | Gmp					(* group-membership properties *)
  | Sync				(* view synchronization *)
  | Total				(* totally ordered messages *)
  | Heal				(* partition healing *)
  | Switch				(* protocol switching *)
  | Auth				(* authentication *)
  | Causal				(* causally ordered broadcasts *)
  | Subcast				(* subcast pt2pt messages *)
  | Frag				(* fragmentation-reassembly *)
  | Debug				(* debugging layers *)
  | Scale				(* scalability *)
  | Xfer				(* state transfer *)
  | Cltsvr				(* client-server management *)
  | Suspect				(* failure detection *)
  | Evs					(* extended virtual synchrony *)
  | Flow        			(* flow control *)
  | Migrate				(* process migration *)
  | Privacy				(* encryption of application data *)
  | Rekey				(* support for rekeying the group *)
  | Primary				(* primary partition detection *)

  | Control				(* attach control panel *)
  | Drop				(* randomized message dropping *)
  | Pbcast				(* Hack: just use pbcast prot. *)

(**************************************************************)

let mapping = [
  Gmp, "Gmp" ;
  Sync, "Sync" ;
  Total, "Total" ;
  Heal, "Heal" ;
  Switch, "Switch" ;
  Auth, "Auth" ;
  Causal, "Causal" ;
  Subcast, "Subcast" ;
  Frag, "Frag" ;
  Debug, "Debug" ;
  Scale, "Scale" ;
  Xfer, "Xfer" ;
  Cltsvr, "Cltsvr" ;
  Suspect, "Suspect" ;
  Evs, "Evs" ;
  Drop, "Drop" ;
  Pbcast, "Pbcast" ;
  Flow, "Flow" ;
  Migrate, "Migrate" ;
  Privacy, "Privacy" ;
  Rekey, "Rekey" ;
  Control, "Control" ;
  Primary, "Primary"
]

let all = List.map fst mapping

let string_of_id id = 
  try List.assoc id mapping with Not_found ->
    failwith "string_of_id:no such id"

let id_of_string s =
  let mapping = List.map (fun (a,b) -> (b,a)) mapping in
  try List.assoc s mapping with Not_found ->
    failwith "id_of_string:no such id"

(**************************************************************)

type r = {
    mutable agree : bool ;
    mutable gmp : bool ;
    mutable sync : bool ;
    mutable total : bool ;
    mutable heal : bool ;
    mutable switch : bool ;
    mutable auth : bool ;
    mutable causal : bool ;
    mutable subcast : bool ;
    mutable frag : bool ;
    mutable debug : bool ;
    mutable scale : bool ;
    mutable xfer : bool ;
    mutable cltsvr : bool ;
    mutable suspect : bool ;
    mutable evs : bool ;
    mutable pbcast : bool ;
    mutable drop : bool ;
    mutable flow : bool ;
    mutable migrate : bool ;
    mutable privacy : bool ;
    mutable rekey : bool ;
    mutable control : bool ;
    mutable primary : bool
} 

(**************************************************************)

let flatten props =
  let r = {
    agree = false ;
    gmp = false ;
    sync = false ;
    total = false ;
    heal = false ;
    switch = false ;
    auth = false ;
    causal = false ;
    subcast = false ;
    frag = false ;
    debug = false ;
    scale = false ;
    xfer = false ;
    cltsvr = false ;
    suspect = false ;
    evs = false ;
    pbcast = false ;
    drop = false ;
    flow = false ;
    migrate = false ;
    privacy = false ;
    rekey = false ;
    control = false ;
    primary = false
  } in
  List.iter (function
  | Agree	-> r.agree <- true
  | Gmp		-> r.gmp <- true
  | Sync	-> r.sync <- true
  | Total	-> r.total <- true
  | Heal	-> r.heal <- true
  | Switch	-> r.switch <- true
  | Auth	-> r.auth <- true
  | Causal	-> r.causal <- true
  | Subcast	-> r.subcast <- true
  | Frag	-> r.frag <- true
  | Debug	-> r.debug <- true
  | Scale	-> r.scale <- true
  | Xfer	-> r.xfer <- true
  | Cltsvr	-> r.cltsvr <- true
  | Suspect	-> r.suspect <- true
  | Evs		-> r.evs <- true
  | Pbcast	-> r.pbcast <- true
  | Drop        -> r.drop <- true
  | Flow        -> r.flow <- true
  | Migrate     -> r.migrate <- true
  | Privacy     -> r.privacy <- true
  | Rekey       -> r.rekey <- true
  | Control     -> r.control <- true
  | Primary     -> r.primary <- true
  ) props ;
  r

(**************************************************************)

let choose props =
  let log = Trace.log name "choose" in
  let p = flatten props in

  let stack =
    if p.pbcast then (
      let stack = 
	["Top"]::
	["Top_appl"]::
	(if p.debug then ["Chk_fifo"] else []) ::
(*
	["Pbcast:Pt2ptw:Pt2pt:Gcast:Bottom"] :: [[]] 
*)
	["Pt2pt:Pbcast:Gcast"] ::
	(if p.drop then ["Drop"] else []) ::
	["Bottom"] :: [[]] 
      in
      let stack = List.flatten stack in
      let stack = String.concat ":" stack in
      stack
    ) else (
      if p.causal && p.total then failwith "Causal and Total not supported" ;
      if p.agree && not p.causal then failwith "Agree requires Causal" ;

      let stack = 
	["Top"] ::
	(if p.control then ["Control"] else []) ::
	(if p.heal then ["Heal"] else []) ::
	(if p.switch then ["Switch"] else []) ::
	(if p.migrate then ["Migrate"] else []) ::
	(if p.rekey then ["Rekey"] else []) :: (* should be above exchagne *)
	(if p.auth then ["Exchange"] else []) ::
	(if p.xfer then ["Xfer"] else []) ::
	(if p.primary then ["Primary"] else []) ::
	(if p.gmp then ["Leave:Inter:Intra:Elect:Merge:Slander"] else []) ::
	(if not (p.sync) then [] else if p.causal then ["TSync"] else ["Sync"]) ::
	(if not (p.suspect) then [] else if p.scale then ["Pr_suspect"] else ["Suspect"]) ::
	(if p.evs then ["Evs"] else []) ::
	(if not p.scale then ["Stable"] else []) ::
	["Top_appl"] ::
	(if p.debug && p.sync then ["Chk_sync"] else []) ::
	(if p.debug then ["Chk_fifo"] else []) ::
	(if p.total then ["Sequencer"] else []) ::
	(if p.privacy then ["Encrypt"] else []) ::
	(if p.frag then ["Frag"] else []) ::

	(* Handle flow control and subcast.  This is a bit
         * confusing to get right.  Basically, people should
         * not use subcast.
	 *)
	[match p.flow, p.subcast with
	| false,false -> "Pt2pt"
	| false, true -> "Subcast"
	| true,false -> "Pt2ptw:Mflow:Pt2pt"
	| true, true -> "Subcast:Window"] ::

	(if p.scale then ["Pr_stable"] else []) ::
	(if p.debug && p.agree then ["Chk_agree"] else []) ::
	(if p.debug && p.causal then ["Chk_hi_causal"] else []) ::
	(if p.agree then ["Agree"] else []) ::
	(if p.debug && p.causal then ["Chk_causal"] else []) ::
	(if p.causal then ["Tcausal"] else ["Mnak"]) ::
(*
      This appears to mess up the iovecs for views of or more members.
*)
	(if p.scale then ["Gcast"] else []) ::
	(if p.drop then ["Drop"] else []) ::
	["Bottom"] :: 
	[[]]
      in
      let stack = List.flatten stack in
      let stack = String.concat ":" stack in
      stack
    ) 
  in
  let stack = Proto.id_of_string stack in
  log (fun () -> sprintf "properties=%s" (string_of_list string_of_id props)) ;
  log (fun () -> sprintf "stack=%s" (Proto.string_of_id stack)) ;
  stack

(**************************************************************)

let vsync = [Gmp;Sync;Heal;Migrate;Switch;Frag;Suspect;Flow]
(*let vsync = [Gmp;Sync;Heal;Switch;Frag;Suspect;Subcast]*)
let total = vsync @ [Total]
let scale = vsync @ [Scale]
let fifo = [Frag]
let transis = [Gmp;Heal;Switch;Migrate;Sync;Suspect;Evs;Subcast;Frag;Causal]
let causal = transis

(**************************************************************)

let strip_groupd properties =
  Lset.subtract properties [Gmp;Sync;Heal;Switch;Suspect]

(**************************************************************)
