(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* LAYER.ML *)
(* Author: Mark Hayden, 4/95 *)
(**************************************************************)
open Util
(**************************************************************)

type 'a saved = 'a option ref

(**************************************************************)

type ('a,'b,'c) handlers_out = {
  up_out	: Event.up -> 'c -> unit ;
  upnm_out	: Event.up -> unit ;
  dn_out 	: Event.dn -> 'c -> 'b -> unit ;
  dnlm_out	: Event.dn -> 'a -> unit ;
  dnnm_out	: Event.dn -> unit
}

type ('a,'b,'c) handlers_in = {
  up_in 	: Event.up -> 'c -> 'b -> unit ;
  uplm_in 	: Event.up -> 'a -> unit ;
  upnm_in 	: Event.up -> unit ;	
  dn_in 	: Event.dn -> 'c -> unit ;
  dnnm_in 	: Event.dn -> unit
}

type ('local,'hdr,'abv) full =
  ('local,'hdr,'abv) handlers_out ->
  ('local,'hdr,'abv) handlers_in

(**************************************************************)
(* What messages look like in this architecture.
 *)
type ('local,'hdr,'abv) msg =
  | NoMsg
  | Local of 'local
  | Full of 'hdr * 'abv
  | Full_nohdr of 'abv
  | Local_nohdr
  | Local_seqno of Trans.seqno
    
(**************************************************************)
(* Layers can specify some headers to optimize against.
 *)
type ('local,'hdr) optimize =
  | NoOpt
  | LocalNoHdr of 'local
  | FullNoHdr of 'hdr
  | LocalSeqno of 'hdr * Event.typ * ('hdr -> Trans.seqno option) * (Trans.seqno -> 'hdr)

(**************************************************************)

type ('bel,'abv) handlers_lout = {
  up_lout	: Event.up -> 'bel -> unit ;
  dn_lout 	: Event.dn -> 'abv -> unit
}

type ('bel,'abv) handlers_lin  = {
  up_lin 	: Event.up -> 'abv -> unit ;
  dn_lin 	: Event.dn -> 'bel -> unit
}

type ('bel,'abv,'state) basic =
    View.full -> 'state * (('bel,'abv) handlers_lout -> ('bel,'abv) handlers_lin)

(**************************************************************)
(* Alias for common layers in this architecture.
 *)
type ('local,'hdr,'state,'abv1,'abv2,'abv3) t = (('abv1,'abv2,'abv3) msg, ('local,'hdr,('abv1,'abv2,'abv3)msg) msg,'state) basic

(**************************************************************)
let compose_msg abv hdr = Full(hdr,abv)
let local_msg hdr = Local(hdr)
let no_msg () = NoMsg
(**************************************************************)

(* Layer conversion functions.
 *)

let hdr_state init hdlrs hdr_t opt args vs =
  (* This check allows us to use '==' below instead of '='.
   *)
  begin 
    match opt with
    | LocalNoHdr hdr ->
	if Obj.is_block (Obj.repr hdr) then
	  failwith "hdr_state:optimization attempted on block header"
    | FullNoHdr hdr ->
	if Obj.is_block (Obj.repr hdr) then
	  failwith "hdr_state:optimization attempted on block header"
    | LocalSeqno(_,typ,_,_) ->
        if Obj.is_block (Obj.repr typ) then
	  failwith "hdr_state:optimization attempted on block header"
    | _ -> ()
  end ;
    
  let state = init args vs in
  let l {up_lout=up;dn_lout=dn} =
    let up ev abv       = up ev abv
    and upnm ev	        = up ev NoMsg
    and dnnm ev         = dn ev NoMsg
    and dn =
      (* Check if we have a header to optimize against. 
       *)
      match opt with
      |	FullNoHdr nohdr ->
	  fun ev abv hdr ->
	    if hdr == nohdr then
              match abv with
	      |	Local_nohdr -> dn ev Local_nohdr
	      |	Local_seqno(seqno) -> dn ev (Local_seqno seqno)
	      |	_ -> dn ev (Full_nohdr abv)
	    else		  
	      dn ev (Full(hdr,abv))
      | LocalSeqno(nohdr,_,det,_) ->
	  fun ev abv hdr ->
	    if hdr == nohdr then (
              match abv with
	      |	Local_nohdr -> dn ev Local_nohdr
	      |	Local_seqno(seqno) -> dn ev (Local_seqno seqno)(*switch*)
	      |	_ -> dn ev (Full_nohdr abv)
	    ) else (
	      match det hdr with
	      |	Some seqno -> (
		  match abv with
		  | Local_nohdr ->
		      dn ev (Local_seqno(seqno))
		  | _ ->
		      dn ev (Full(hdr,abv))
	        )
	      |	None ->
		  dn ev (Full(hdr,abv))
	    )
      |	_ -> 
	  fun ev abv hdr ->
	    dn ev (Full(hdr,abv))
    and dnlm =
      match opt with
      |	LocalNoHdr nohdr ->
	  fun ev hdr ->
	    if hdr == nohdr then
	      dn ev Local_nohdr
	    else dn ev (Local hdr) 
      | _ ->
	  fun ev hdr ->
	    dn ev (Local hdr)
    in
    
    let {up_in=up;uplm_in=uplm;upnm_in=upnm;dn_in=dn;dnnm_in=dnnm} =
      hdlrs state vs {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} 
    in

    let up =
      match opt with
      |	FullNoHdr nohdr ->
	  (fun ev msg -> 
	    match msg with
	    | Local_seqno seq -> up ev (Local_seqno seq) nohdr
	    | Local_nohdr -> up ev Local_nohdr nohdr
	    | Full_nohdr abv -> up ev abv nohdr
	    | Local hdr     -> uplm ev     hdr
	    | Full(hdr,abv) -> up   ev abv hdr
	    | NoMsg	      -> upnm ev)
      |	LocalNoHdr nohdr ->
	  (fun ev msg -> 
	    match msg with
	    | Local_nohdr   -> uplm ev nohdr
	    | Local hdr     -> uplm ev     hdr
	    | Full(hdr,abv) -> up   ev abv hdr
	    | NoMsg	    -> upnm ev
	    | _ -> failwith "hdr_state:LocalNoHdr:sanity")
      | LocalSeqno(nohdr,typ,_,cons) ->
	  (fun ev msg ->
	    match msg with
	    | Local_seqno seq -> 
		if Event.getType ev == typ then
		  up ev Local_nohdr (cons seq)
		else 
		  up ev (Local_seqno seq) nohdr
	    | Local_nohdr -> up ev Local_nohdr nohdr
	    | Full_nohdr abv -> up ev abv nohdr
	    | Local hdr     -> uplm ev     hdr
	    | Full(hdr,abv) -> up   ev abv hdr
	    | NoMsg	    -> upnm ev
(*
	    | _ -> failwith "hdr_state:LocalSeqno:sanity"
*))
      |	NoOpt ->
	  (fun ev msg -> 
	    match msg with
	    | Local hdr     -> uplm ev     hdr
	    | Full(hdr,abv) -> up   ev abv hdr
	    | NoMsg	    -> upnm ev
	    | _ -> failwith "hdr_state:NoOpt:sanity")
    and dn ev abv =
      match abv with
      | NoMsg -> dnnm ev
      | _ -> dn ev abv
    in

    {up_lin=up;dn_lin=dn}
  in (state,l)

(**************************************************************)

let hdr init hdlr hdr_t opt args vs =
  let (_,l) = hdr_state init hdlr hdr_t opt args vs in
  ((),l)

(**************************************************************)
(* Layer conversion function with no optimizations.
 *)

let hdr_state_noopt init hdlrs args vs =
  let state = init args vs in
  let l {up_lout=up;dn_lout=dn} =
    let up ev abv       = up ev abv
    and upnm ev	        = up ev NoMsg
    and dnnm ev         = dn ev NoMsg
    and dn ev abv hdr   = dn ev (Full(hdr,abv))
    and dnlm ev hdr     = dn ev (Local hdr)
    in
    
    let {up_in=up;uplm_in=uplm;upnm_in=upnm;dn_in=dn;dnnm_in=dnnm} =
      hdlrs state vs {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} 
    in

    let up ev msg = match msg with
    | Local hdr     -> uplm ev     hdr
    | Full(hdr,abv) -> up   ev abv hdr
    | NoMsg	    -> upnm ev
    | _ -> failwith "hdr_state_noopt:sanity"
    and dn ev abv = match abv with
    | NoMsg         -> dnnm ev
    | _             -> dn ev abv
    in

    {up_lin=up;dn_lin=dn}
  in (state,l)

(**************************************************************)
(* Record containing overall fields for protocol stacks.
 * Assumes no layers are repeated.  
 *)

type state = {
  interface     : Appl_intf.t ;
  switch	: Time.t saved
}

let new_state interface = {
  interface     = interface ;
  switch	= ref None
}

let reset_state s =
  s.switch := None

(**************************************************************)

type lyr = (unit,unit,unit) basic

let table = 
  let table = Hashtbl.create 10 in
  Trace.install_root (fun () ->
    [sprintf "LAYER:#layers=%d" (hashtbl_size table)]
  ) ;
  table

let install name layer = 
  let name = String.uppercase name in
  Hashtbl.add table name layer

let lookup name = 
  let name = String.uppercase name in
  Hashtbl.find table name

(**************************************************************)

let magic o = Obj.magic (Obj.repr o)

type ('args,'b,'c,'d) init = 
  ('args -> ('b  ,'c  ,'d  ) basic) -> 
  state -> (unit,unit,unit) basic

let coerce (l : ('a,'b,'c)basic) = (magic l : lyr)

let init l s         = coerce (l ())
let initSwitch l s   = coerce (l s.switch)
let initTop_appl l s = coerce (l (s.interface))
let initNamed l s    = coerce (l "???")

(**************************************************************)

let procurers = ref ["internal_table",lookup]

let finder name lookup =
  procurers := !procurers @ [name,lookup]

let procure_hide name state =
  let rec loop = function
  | [] ->
      eprintf "LAYER:couldn't find layer:%s\n" name ;
      failwith "no such layer"
  | (_,lookup) ::tl ->
      try
	let l = lookup name in
	let l = (l : state -> lyr) in
	let l = magic l in
	let l = (l : state -> ('a,'b,'c) basic) in
	l state
      with Not_found -> (
	loop tl
      )
  in loop !procurers

let look_again name =
  let l = (lookup name : state -> lyr) in
  let l = (magic l : state -> ('a,'b,'c) basic) in
  l

(**************************************************************)

let config_print () =
  eprintf "  Procure:Installed Layers\n" ;
  Hashtbl.iter (fun name _ ->
    eprintf "	%s\n" name
  ) table

(**************************************************************)
