(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* EVENT.MLI *)
(* Author: Mark Hayden, 4/95 *)
(* Based on Horus events by Robbert vanRenesse *)
(**************************************************************)
open Trans
(**************************************************************)

  (* Up and down events *)
type t
type up = t
type dn = t

(**************************************************************)

(* 
 * ACKNOWLEDGEMENT: fields of this type appear in both up
 * and down events.  The default value for an acknowledgement
 * is NoAck.  The usual way acknowledgements work is for
 * some layer to set the ack field in an up event with an
 * acknowledgement that it wants the event to be
 * acknowledged with.  The layer which initiate the message
 * then is responsible for later delivering down a EAck with
 * the ack field set to be the same acknowledgment.  
 *)

type acknowledgement =
  | NoAck
  | RankSeqno of origin * seqno

val string_of_ack : acknowledgement -> string

(**************************************************************)

  (* Event types *)
type typ =
    (* These events should have messages associated with them. *)
  | ECast				(* Multicast message *)
  | ESend				(* Pt2pt message *)
  | EMergeRequest			(* Request a merge *)
  | EMergeGranted			(* Grant a merge request *)
  | EOrphan				(* Message was orphaned *)

    (* These types do not have messages. *)
  | EAccount				(* Output accounting information *)
  | EAck				(* Acknowledge message *)
  | EAlive				(* added by Ohad *)
  | EAsync				(* Asynchronous application event *)
  | EBlock				(* Block the group *)
  | EBlockOk				(* Acknowledge blocking of group *)
  | EDump				(* Dump your state (debugging) *)
  | EElect				(* I am now the coordinator *)
  | EExit				(* Disable this stack *)
  | EFail				(* Fail some members *)
  | EGossipExt				(* Gossip message *)
  | EInit				(* First event delivered *)
  | EInvalid				(* Erroneous event type *)
  | ELeave				(* A member wants to leave *)
  | ELostMessage			(* Member doesn't have a message *)
  | EMergeDenied			(* Deny a merge request *)
  | EMergeFailed			(* Merge request failed *)
  | EMigrate				(* Change my location *)
  | EPrompt				(* Prompt a new view *)
  | EProtocol				(* Request a protocol switch *)
  | ERekey				(* Request a rekeying of the group *)
  | EStable				(* Deliver stability down *)
  | EStableReq				(* Request for stability information *)
  | ESuspect				(* Member is suspected to be faulty *)
  | ESystemError			(* Something serious has happened *)
  | ETimer				(* Request a timer *)
  | ETransView				(* added by Ohad *)
  | EView				(* Notify that a new view is ready *)
  | EXferDone				(* Notify that a state transfer is complete *)

  | EInfo				(* Management info (for Alexey) *)

(**************************************************************)

type field =
      (* Common fields *)
  | Type	of typ			(* type of the message*)
  | Origin	of rank			(* rank of the sender *)
  | Ack	        of acknowledgement	(* acknowledgement information *)
  | Ranks       of rank list		(* list of ranks (usually dests) *)
  | Iov	        of Iovecl.t		(* payload of message *)

      (* Uncommon fields *)
  | Address     of Addr.set		(* new address for a member *)
  | Failures	of rank list		(* failed members *)
  | Suspects	of rank list            (* suspected members *)
  | SuspectReason of string		(* reasons for suspicion *)
  | Stability	of seqno array		(* stability vector *)
  | NumCasts	of seqno array		(* number of casts seen *)
  | Mergers	of View.state		(* list of merging members *)
  | Contact	of Endpt.full * View.id option (* contact for a merge *)
  | HealGos	of Proto.id * View.id * Endpt.full * View.t (* HEAL gossip *)
  | SwitchGos	of Proto.id * View.id * Time.t  (* SWITCH gossip *)
  | ExchangeGos	of string		(* EXCHANGE gossip *)
  | ViewState	of View.state		(* state of next view *)
  | ProtoId	of Proto.id		(* protocol id (only for down events) *)
  | Time        of Time.t		(* current time *)
  | Alarm       of Time.t		(* for alarm requests *)
  | Control     of (string * string)list (* for Control layer (for Alexey) *)

      (* Flags *)
  | Unreliable				(* message is unreliable *)
  | NoTotal				(* message is not totally ordered*)
  | ServerOnly				(* deliver only at servers *)
  | ClientOnly				(* deliver only at clients *)

      (* Debugging *)
  | ApplMsg				(* was this message generated by an appl? *)
  | History	of string		(* debugging history *)
      
      (* TransisE *)
  | Causal				(* causal ordering *)
  | Agreed				(* agreed ordering *)
  | Safe				(* safe delivery *)
  | AckVct        of seqno option array
  | Transitional  of bool array

      (* A hack... *)
  | Forcecopy

(**************************************************************)

  (* Directed events. *)
type dir = Up of t | Dn of t

type ('a,'b) dirm = UpM of up * 'a | DnM of dn * 'b

(**************************************************************)

  (* Constructor *)
val create	: debug -> typ -> field list -> t

(**************************************************************)

  (* Destructor *)
val free	: debug -> t -> unit

(**************************************************************)

  (* Copier *)
val copy	: debug -> t -> t
  
(**************************************************************)

  (* Modifier *)
val set		: debug -> t -> field list -> t

(**************************************************************)

  (* Sanity checkers *)
val upCheck	: debug -> t -> unit
val dnCheck	: debug -> t -> unit

(**************************************************************)

  (* Pretty printer *)
val to_string	: t -> string
val string_of_type : typ -> string

(**************************************************************)

  (* Special constructors *)

val bodyCore		: debug -> typ -> origin -> Iovecl.t -> t
val bypassCast		: debug -> origin -> Iovecl.t -> t
val bypassSend		: debug -> origin -> Iovecl.t -> t
val castEv		: debug	-> t
val castIov		: debug -> Iovecl.t -> t
val castOriginIov	: debug -> rank -> Iovecl.t -> t
val sendRank		: debug -> rank	-> t
val sendRanks		: debug -> rank list -> t
val sendRanksIov	: debug -> rank list -> Iovecl.t -> t
val suspectReason       : debug -> rank list -> debug -> t
val timerAlarm		: debug -> Time.t -> t
val timerTime		: debug -> Time.t -> t

(**************************************************************)

  (* Accessors *)

val getAck		: t -> acknowledgement
val getAddress     	: t -> Addr.set
val getAlarm		: t -> Time.t
val getApplMsg		: t -> bool
val getClientOnly	: t -> bool
val getControl          : t -> (string * string) list
val getContact		: t -> Endpt.full * View.id option
val getExtend     	: t -> field list
val getExtender     	: (field -> 'a option) -> 'a -> t -> 'a
val getExtendOpt	: t -> (field -> bool) -> unit
val getFailures		: t -> rank list
val getIov		: t -> Iovecl.t
val getMergers		: t -> View.state
val getNoTotal		: t -> bool
val getNumCasts		: t -> seqno array
val getOrigin		: t -> origin
val getProtoId		: t -> Proto.id
val getRanks		: t -> rank list
val getServerOnly	: t -> bool
val getStability	: t -> seqno array
val getSuspectReason	: t -> string
val getSuspects		: t -> rank list
val getTime		: t -> Time.t
val getType		: t -> typ
val getUnreliable	: t -> bool
val getViewState	: t -> View.state

  (* TransisE additions *)
val getAckVct	      : t -> seqno option array
val getTransitional   : t -> bool array

(**************************************************************)

val getIovLen           : t -> len

(**************************************************************)

  (* For automating acknowledgements *)
val make_acker	: string -> (t -> unit) -> t -> unit

(**************************************************************)
