(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* EVENT.ML *)
(* Author: Mark Hayden, 4/95 *)
(* Based on Horus events by Robbert vanRenesse *)
(**************************************************************)
open Util
open Trans
(**************************************************************)
let name = Trace.source_file "EVENT"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type acknowledgement =
  | NoAck
  | RankSeqno of origin * seqno

(**************************************************************)

type typ =
    (* These events should have messages associated with them. *)
  | ECast				(* Multicast message *)
  | ESend				(* Pt2pt message *)
  | EMergeRequest			(* Request a merge *)
  | EMergeGranted			(* Grant a merge request *)
  | EOrphan				(* Message was orphaned *)

    (* These types do not have messages. *)
  | EAccount				(* Output accounting information *)
  | EAck				(* Acknowledge message *)
  | EAlive				(* added by Ohad *)
  | EAsync				(* Asynchronous application event *)
  | EBlock				(* Block the group *)
  | EBlockOk				(* Acknowledge blocking of group *)
  | EDump				(* Dump your state (debugging) *)
  | EElect				(* I am now the coordinator *)
  | EExit				(* Disable this stack *)
  | EFail				(* Fail some members *)
  | EGossipExt				(* Gossip message *)
  | EInit				(* First event delivered *)
  | EInvalid				(* Erroneous event type *)
  | ELeave				(* A member wants to leave *)
  | ELostMessage			(* Member doesn't have a message *)
  | EMergeDenied			(* Deny a merge request *)
  | EMergeFailed			(* Merge request failed *)
  | EMigrate				(* Change my location *)
  | EPrompt				(* Prompt a new view *)
  | EProtocol				(* Request a protocol switch *)
  | ERekey				(* Request a rekeying of the group *)
  | EStable				(* Deliver stability down *)
  | EStableReq
  | ESuspect				(* Member is suspected to be faulty *)
  | ESystemError			(* Something serious has happened *)
  | ETimer				(* Request a timer *)
  | ETransView				(* added by Ohad *)
  | EView				(* Notify that a new view is ready *)
  | EXferDone				(* Notify that a state transfer is complete *)

  | EInfo				(* Management info (for Alexey) *)

(**************************************************************)

type field =
      (* Common fields *)
  | Type	of typ			(* type of the message*)
  | Origin	of rank			(* rank of the sender *)
  | Ack	        of acknowledgement	(* acknowledgement information *)
  | Ranks       of rank list		(* list of ranks (usually dests) *)
  | Iov	        of Iovecl.t		(* payload of message *)

      (* Uncommon fields *)
  | Address     of Addr.set		(* new address for a member *)
  | Failures	of rank list		(* failed members *)
  | Suspects	of rank list            (* suspected members *)
  | SuspectReason of string		(* reasons for suspicion *)
  | Stability	of seqno array		(* stability vector *)
  | NumCasts	of seqno array		(* number of casts seen *)
  | Mergers	of View.state		(* list of merging members *)
  | Contact	of Endpt.full * View.id option (* contact for a merge *)
  | HealGos	of Proto.id * View.id * Endpt.full * View.t (* HEAL gossip *)
  | SwitchGos	of Proto.id * View.id * Time.t  (* SWITCH gossip *)
  | ExchangeGos	of string		(* EXCHANGE gossip *)
  | ViewState	of View.state		(* state of next view *)
  | ProtoId	of Proto.id		(* protocol id (only for down events) *)
  | Time        of Time.t		(* current time *)
  | Alarm       of Time.t		(* for alarm requests *)
  | Control     of (string * string)list (* for Control layer (for Alexey) *)

      (* Flags *)
  | Unreliable				(* message is unreliable *)
  | NoTotal				(* message is not totally ordered*)
  | ServerOnly				(* deliver only at servers *)
  | ClientOnly				(* deliver only at clients *)

      (* Debugging *)
  | ApplMsg				(* was this message generated by an appl? *)
  | History	of string		(* debugging history *)
      
      (* TransisE *)
  | Causal				(* causal ordering *)
  | Agreed				(* agreed ordering *)
  | Safe				(* safe delivery *)
  | AckVct        of seqno option array
  | Transitional  of bool array

      (* A hack... *)
  | Forcecopy

(**************************************************************)

type t = {
  ty		: typ ;
  origin	: rank ;
  ranks         : rank list ;
  ack	        : acknowledgement ;
  iov	        : Iovecl.t ;
  extend	: field list
}

type up = t
type dn = t

(**************************************************************)

type dir = Up of t | Dn of t

type ('a,'b) dirm = UpM of t * 'a | DnM of t * 'b

(**************************************************************)

let invalid_rank = -1

(**************************************************************)

let default = {
  ty 		= EInvalid ;
  origin 	= invalid_rank ;
  ranks         = [] ;
  ack	        = NoAck ;
  iov	        = [||] ;
  extend 	= []
}

(**************************************************************)

let bodyCore debug ty origin iov = {
  ty 		= ty ;
  origin	= origin ;
  ranks         = [] ;
  iov	        = iov ;
  ack	        = NoAck ;
  extend 	= []
}

(**************************************************************)

let bypassCast debug origin iov = {
  ty 		= ECast ;
  origin 	= origin ;
  ranks         = [] ;
  iov	        = iov ;
  ack	        = NoAck ;
  extend 	= []
}

(**************************************************************)

let bypassSend debug origin iov = {
  ty 		= ESend ;
  origin 	= origin ;
  ranks         = [] ;
  iov	        = iov ;
  ack	        = NoAck ;
  extend 	= []
}

(**************************************************************)

let castIov debug iov = {
  ty 		= ECast ;
  ranks		= [] ;
  ack 	        = NoAck ;
  iov	        = iov ;
  extend	= [] ;
  origin        = invalid_rank
}

(**************************************************************)

let rec setLoop ran ack iov ext ori typ usl = match usl with
  | [] -> {origin=ori;extend=ext;ack=ack;ty=typ;iov=iov;ranks=ran}
  | hd :: tl -> match hd with
    | Type   typ -> setLoop ran ack iov ext ori typ tl
    | Origin ori -> setLoop ran ack iov ext ori typ tl
    | Ranks  ran -> setLoop ran ack iov ext ori typ tl
    | Ack    ack -> setLoop ran ack iov ext ori typ tl
    | Iov    iov -> setLoop ran ack iov ext ori typ tl
    | Forcecopy  -> setLoop ran ack iov ext ori typ tl
    | f          -> setLoop ran ack iov (f::ext) ori typ tl

let set debug {ranks=ran;ack=ack;ty=typ;iov=iov;origin=ori;extend=ext} dsl =
  setLoop ran ack iov ext ori typ dsl

let create =
  let partial_eval = setLoop 
    default.ranks
    default.ack
    default.iov
    default.extend
    default.origin
  in
  fun debug typ fields -> partial_eval typ fields

let free debug ev = Iovecl.free debug ev.iov

let copy debug ev = Array.map (Iovec.ref debug) ev.iov ; ev

let upCheck debug ev = Array.map (Iovec.check debug) ev.iov ; ()

let dnCheck debug ev = Array.map (Iovec.check debug) ev.iov ; ()

(**************************************************************)

let getExtender f g =
  let rec loop = function
    | [] -> g
    | h::t ->
	match f h with
	| Some o -> o
	| None -> loop t
  in fun e -> loop e.extend

let getExtendFail f field =
  let rec loop = function
    | [] -> failwith (sprintf "get%s:no such field" field)
    | h::t ->
	match f h with
	| Some o -> o
	| None -> loop t
  in fun e -> loop e.extend

let getExtendOpt e f =
  let rec loop = function
  | [] -> ()
  | h::t ->
     if not (f h) then loop t
  in loop e.extend

let getAck e    = e.ack
let getExtend e = e.extend
let getIov e    = e.iov
let getOrigin e = e.origin
let getRanks e  = e.ranks
let getType e   = e.ty

let getAckVct = getExtendFail (function (AckVct a) -> Some a | _ -> None) "AckVct"
let getAlarm = getExtendFail (function (Alarm i) -> Some i | _ -> None) "Alarm"
let getContact = getExtendFail (function (Contact(i,j)) -> Some(i,j) | _ -> None) "Contact"
let getFailures = getExtendFail (function (Failures i) -> Some i | _ -> None) "Failures"
let getMergers = getExtendFail (function (Mergers i) -> Some i | _ -> None) "Mergers"
let getAddress = getExtendFail (function (Address i) -> Some i | _ -> None) "Address"
let getNumCasts = getExtendFail (function (NumCasts i) -> Some i | _ -> None) "NumCasts"
let getProtoId = getExtendFail (function (ProtoId i) -> Some i | _ -> None) "ProtoId"
let getStability = getExtendFail (function (Stability i) -> Some i | _ -> None) "Stability"
let getSuspectReason = getExtendFail (function (SuspectReason i) -> Some i | _ -> None) "SuspectReason"
let getSuspects = getExtendFail (function (Suspects i) -> Some i | _ -> None) "Suspects"
let getTime = getExtendFail (function (Time i) -> Some i | _ -> None) "Time"
let getTransitional = getExtendFail (function (Transitional a) -> Some a | _ -> None) "Transitional"
let getViewState = getExtendFail (function (ViewState l) -> Some l | _ -> None) "ViewState"
let getControl = getExtender (function (Control i) -> Some i | _ -> None) []

let getApplMsg ev = match ev.extend with [] -> false | ext -> List.memq ApplMsg ext
let getClientOnly ev = match ev.extend with [] -> false | ext -> List.memq ClientOnly ext
let getUnreliable ev = match ev.extend with [] -> false | ext -> List.memq Unreliable ext
let getServerOnly ev = match ev.extend with [] -> false | ext -> List.memq ServerOnly ext
let getNoTotal ev = match ev.extend with [] -> false | ext -> List.memq NoTotal ext

(**************************************************************)

let getIovLen ev =
  let iov = getIov ev in
  Iovecl.len name iov

(**************************************************************)

let string_of_ack = function
  | NoAck -> "NoAck"
  | RankSeqno(from,seqno) -> 
      sprintf "RankSeqno(%d,#%d)" from seqno
      
(**************************************************************)

let string_of_type = function
  | EAccount	    	-> "EAccount"
  | EAck	    	-> "EAck"
  | EAlive              -> "EAlive"
  | EAsync	  	-> "EAsync"
  | EBlock	  	-> "EBlock"
  | EBlockOk	  	-> "EBlockOk"
  | ECast	  	-> "ECast"
  | EDump	  	-> "EDump"
  | EElect		-> "EElect"
  | EExit	  	-> "EExit"
  | EFail	  	-> "EFail"
  | EGossipExt		-> "EGossipExt"
  | EInit	  	-> "EInit"
  | EInfo               -> "EInfo"
  | EInvalid	  	-> failwith "string_of_type:EInvalid"
  | ELeave	    	-> "ELeave"
  | ELostMessage 	-> "ELostMessage"
  | EMergeRequest 	-> "EMergeRequest"
  | EMergeDenied 	-> "EMergeDenied"
  | EMergeFailed 	-> "EMergeFailed"
  | EMergeGranted  	-> "EMergeGranted"
  | EMigrate            -> "EMigrate"
  | EOrphan	  	-> "EOrphan"
  | EPrompt             -> "EPrompt"
  | EProtocol		-> "EProtocol"
  | ERekey              -> "ERekey"
  | ESend	  	-> "ESend"
  | EStable	  	-> "EStable"
  | EStableReq	  	-> "EStableReq"
  | ESuspect	  	-> "ESuspect"
  | ESystemError 	-> "ESystemError"
  | ETimer	  	-> "ETimer"
  | ETransView          -> "ETransView"
  | EView	  	-> "EView"
  | EXferDone		-> "EXferDone"

(**************************************************************)

let string_of_field f g name v = if f v then Some(name^(g v)) else None
let int_field_neg   = string_of_field (fun i -> i >= 0) string_of_int
let int_list_field  = string_of_field (fun v -> v <> []) string_of_int_list
let ack_field       = string_of_field (fun a -> a <> NoAck) string_of_ack

(**************************************************************)

let string_of_fields = function
| Failures	e -> ("Failures="^( string_of_int_list e))
| Suspects	e -> ("Suspects="^( string_of_int_list e))
| SuspectReason e -> ("SuspectReason="^( e))
| Stability	e -> ("Stability="^( string_of_int_array e))
| NumCasts	e -> ("NumCasts="^( string_of_int_array e))
| Mergers	e -> ("Mergers="^( ""(*Endpt.string_of_id_list e*)))
| Contact	(e,f) -> ("Contact="^( Endpt.string_of_full e))
| HealGos	(e,f,g,h) -> ("HealGos="^( "HealGos"))
| SwitchGos	(e,f,g) -> ("SwitchGos="^( "SwitchGos"))
| ExchangeGos	e -> ("ExchangeGos="^( "ExchangeGos"))
| History	e -> ("History="^( e))
| ViewState	e -> ("ViewState="^(View.string_of_state e))
| ProtoId	e -> ("ProtoId="^( Proto.string_of_id e))
| AckVct        e -> ("AckVct="^("unimp"))
| ApplMsg         -> ("ApplMsg")
| Transitional  e -> ("Transitional="^("unimp"))
| Unreliable      -> ("Unreliable")
| NoTotal         -> ("NoTotal")
| ClientOnly      -> ("ClientOnly")
| ServerOnly      -> ("ServerOnly")
| Time          e -> ("Time="^(Time.to_string e))
| Alarm         e -> ("Alarm="^(Time.to_string e))
| Causal          -> ("Causal")
| Agreed          -> ("Agreed")
| Safe            -> ("Safe")
| Control i         -> (sprintf "Control=%s" (string_of_list (fun (k,d) -> sprintf "%s=%s" k d) i))
| _ -> failwith "unknown field"

let to_string ev =
  let l = (filter_nones [
    int_field_neg   "origin=" ev.origin ;
    int_list_field  "ranks="    ev.ranks ;
    ack_field	    "ack="    ev.ack
  ]) @ (List.map string_of_fields ev.extend) in
  sprintf "%s{%s}" 
    (string_of_type ev.ty) 
    (String.concat ":" l)

(**************************************************************)

let castEv	  debug	     = create debug ECast[]
let castOriginIov debug o i  = create debug ECast[Origin o;Iov i]
let sendRank      debug d    = create debug ESend[Ranks [d]]
let sendRanks	  debug ds   = create debug ESend[Ranks ds]
let sendRanksIov  debug ds i = create debug ESend[Ranks ds; Iov i]
let suspectReason debug s r  = create debug ESuspect[Suspects s;SuspectReason r]
let timerAlarm    debug t    = create debug ETimer[(Alarm t)]
let timerTime     debug t    = create debug ETimer[Time t]

(**************************************************************)

let make_acker debug dnnm = function
  | {ack=NoAck} -> ()
  | {ack=ack} -> dnnm (create debug EAck[Ack ack])

(**************************************************************)
