(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* DOMAIN.MLI *)
(* Author: Mark Hayden, 4/96 *)
(**************************************************************)
open Trans
(**************************************************************)

(* DOMAIN.T: The type of a communication domain.
 *)

type t

(* DEST: this specifies the destinations to use in xmit and
 * xmitv.  For pt2pt xmits, we give a list of endpoint
 * identifiers to which to send the message.  For
 * multicasts, we give a group address and a boolean value
 * specifying whether loopback is necessary.  If no loopback
 * is requested then the multicast may not be delivered to
 * any other processes on the same host as the sender.
 *)

(* Note that significant preprocessing may be done on the
 * destination values.  For optimal performance, apply the
 * destination to an xmit outside of your critical path.
 *)

type loopback = bool

type dest =
| Pt2pt of Addr.set array
| Mcast of Group.id * loopback
| Gossip of Group.id

(**************************************************************)
(* Operations on domains.
 *)

val name 	: t -> string
val enable 	: t -> Addr.id -> Group.id -> Addr.set -> View.t -> unit
val disable 	: t -> Addr.id -> Group.id -> Addr.set -> View.t -> unit
val xmit 	: t -> Addr.id -> dest ->
  ((buf -> ofs -> len -> unit) * (Iovecl.t -> unit)) option
val addr	: t -> Addr.id -> Addr.t

(* Create a domain.
 *)
val create : 
  name ->				(* name of domain *)
  (Addr.id -> Addr.t) ->		(* create endpoint address *)
  (Addr.id -> Group.id -> Addr.set -> View.t -> unit) -> (* enable new transport *)
  (Addr.id -> Group.id -> Addr.set -> View.t -> unit) -> (* disable transport *)
  (Addr.id -> dest ->			(* xmit *)
    ((buf -> ofs -> len -> unit) * 
     (Iovecl.t -> unit)) option) ->
  t

val string_of_dest : dest -> string

(**************************************************************)
(* Look up a domain in the domain table.
 *)

val of_mode : Addr.id -> t
val install : Addr.id -> (Alarm.t -> t) -> unit

(**************************************************************)
