(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* DOMAIN.ML *)
(* Author: Mark Hayden, 4/96 *)
(**************************************************************)
open Util
open Trans
(**************************************************************)
let name = Trace.source_file "DOMAIN"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type loopback = bool

type dest =
| Pt2pt of Addr.set array
| Mcast of Group.id * loopback
| Gossip of Group.id

type t = {
  name		: string ;
  addr      	: Addr.id -> Addr.t ;
  enable    	: Addr.id -> Group.id -> Addr.set -> View.t -> unit ;
  disable 	: Addr.id -> Group.id -> Addr.set -> View.t -> unit ;
  xmit		: Addr.id -> dest -> 
    ((buf -> ofs -> len -> unit) * 
     (Iovecl.t -> unit)) option
}

(**************************************************************)

let string_of_dest = function
| Pt2pt d    -> sprintf "{Pt2pt:%s}"    (string_of_array Addr.string_of_set d)
| Mcast(d,l) -> sprintf "{Mcast:%s:%b}" (Group.string_of_id d) l
| Gossip(d)  -> sprintf "{Gossip:%s}"   (Group.string_of_id d)

let name    d = d.name
let enable  d = d.enable
let disable d = d.disable
let xmit    d = d.xmit
let addr    d = d.addr

let create name addr enable disable xmit = {
  name 	  	= name ;
  addr 		= addr ;
  enable  	= enable ;
  disable 	= disable ;
  xmit		= xmit
}

(**************************************************************)
(* Domain management.
 *)

let dormant = ref []
let activated = ref []

let install id d =
  dormant := (id,d) :: !dormant

let of_mode mode =
  (* Use the Udp domain for Deering.
   *)
  let mode = match mode with
  | Addr.Deering -> Addr.Udp
  | Addr.Sp2 -> Addr.Udp
  | _ -> mode
  in

  (* If in activated list, use that.
   *)
  try 
    List.assoc mode !activated 
  with Not_found ->
    try
      (* Look up in dormant list.
       *)
      let d = List.assoc mode !dormant in

      (* Remove it from dormant list.
       *)
      dormant := 
        List.fold_left (fun o ((i,_) as it) ->
	  if mode = i then o else it :: o
        ) [] !dormant ;

      (* Enable the domain and add to activated list.
       *)
      let d = d (Alarm.get ()) in
      activated := (mode,d) :: !activated ;
      d
    with Not_found ->
      eprintf "DOMAIN:domain '%s' not installed, exiting\n"
      (Addr.string_of_id mode) ;
      exit 1

(**************************************************************)
