(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* CONN.ML : communication connection ids *)
(* Author: Mark Hayden, 12/95 *)
(**************************************************************)
open Util
open Trans
open Hsys
(**************************************************************)
let name = Trace.source_file "CONN"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type sndr_mbr = rank			(* for members *)
type dest_mbr = rank			(* for members *)
type dest_endpt = Endpt.id		(* for merging *)

(**************************************************************)

type id = {
  version       : Version.id ;
  group 	: Group.id ;
  stack 	: Stack_id.t ;
  proto 	: Proto.id ;
  view_id 	: View.id option ;
  sndr_mbr 	: sndr_mbr ;
  dest_mbr 	: dest_mbr ;
  dest_endpt 	: dest_endpt option
}

type kind = Cast | Send | Other

(**************************************************************)

let string_of_kind = function
| Cast -> "Cast"
| Send -> "Send"
| Other -> "Other"

(**************************************************************)

let int_of_mbr = function
| Some mbr ->
    if mbr = -1 then failwith "sanity" ;
    mbr
| None -> (-1)

(**************************************************************)

let mbr_of_int = function
| -1 -> None
|  i -> Some i

(**************************************************************)

let squash_sender id = 
  let sender = mbr_of_int id.sndr_mbr in
  let id = {
    version = id.version ;
    group = id.group ;
    stack = id.stack ;
    proto = id.proto ;
    view_id = id.view_id ;
    sndr_mbr = 0 ;			(* hack! (should be -2) *)
    dest_mbr = id.dest_mbr ;
    dest_endpt = id.dest_endpt
  } in (sender,id)

(**************************************************************)

let id version group stack proto view_id sndr_mbr dest_mbr dest_endpt = {  
  version       = version ;
  group 	= group ;
  stack 	= stack ;
  proto 	= proto ;
  view_id 	= view_id ;
  sndr_mbr 	= int_of_mbr sndr_mbr ;
  dest_mbr 	= int_of_mbr dest_mbr ;
  dest_endpt	= dest_endpt
}

(**************************************************************)

let marsh,_ = Util.make_marsh name false

let hash_of_id id =
  let id = Obj.repr id in
  let id = Deepcopy.f id in
  let id = marsh id in
  Digest.string id
  
(*
  (* Force the first word to be properly "pushed".
   *)
  let word1 = String.sub id 0 4 in
  let word1 = Hashtbl.hash word1 in
  push_int id 0 word1 ;
*)

(**************************************************************)

let string_of_id id =
  let (view_id,sndr_mbr,dest_mbr,dest_endpt) =
    id.view_id,id.sndr_mbr,id.dest_mbr,id.dest_endpt
  in
  let kind = match id.view_id,mbr_of_int id.sndr_mbr,mbr_of_int id.dest_mbr,id.dest_endpt with
    | Some _, Some _, Some _, None   -> "pt2pt"
    | Some _, Some _, None  , None   -> "cast"
    | Some _, None  , None  , Some _ -> "merge"
    | None  , None  , None  , None   -> "gossip"
    | _ -> "?" 
  in
  sprintf "{Conn:%s:%s:%s:%s:%s:from=%s:to=%s/%s}"
    kind
    (Group.string_of_id id.group)
    (Stack_id.string_of_id id.stack)
    (Proto.string_of_id id.proto)
    (string_of_option View.string_of_id id.view_id)
    (string_of_option string_of_int (mbr_of_int id.sndr_mbr))
    (string_of_option string_of_int (mbr_of_int id.dest_mbr))
    (string_of_option Endpt.string_of_id id.dest_endpt)

(**************************************************************)

(* Set of connection identifiers being used to communicate
 * with peers.
 *)
type id_field = Version.id * Endpt.id * Group.id * View.id * Stack_id.t * Proto.id
type t = {
  id	     	: id_field ;

  pt2pt_send 	: id array ;
  pt2pt_recv	: id array ;

  multi_send 	: id ;
  multi_recv	: id array ;

  merge_send 	: View.id option -> Endpt.id -> id ;
  merge_recv	: id ;

  gossip 	: id ;

  all_recv 	: (id * kind * rank * string) list
}

(**************************************************************)

let string_of_id_field (version,endpt,group,view_id,stack,proto) =
  sprintf "{Conn.t.id:ver=%s;endpt=%s;group=%s;view_id=%s;stack=%s;proto=%s}" 
    (Version.string_of_id version)
    (Endpt.string_of_id endpt)
    (Group.string_of_id group)
    (View.string_of_id view_id)
    (Stack_id.string_of_id stack)
    (Proto.string_of_id proto)

let string_of_t t = (string_of_id_field t.id)

(**************************************************************)
(* Create a record of connections for an id & view.
 *)

let s_gossip = "Gossip"
let s_merge = "Merge"
let s_send = "Send"
let s_cast = "Cast"

let create version endpt group view_id stack_id proto_id view gossip_trans =
  let nmembers = Array.length view in
  let my_rank = array_index endpt view in
  let ranks = Array.of_list (Util.sequence nmembers) in
  let array f = Array.map f ranks in
  let id = id version group stack_id proto_id in

  let multi_send = 
    id (Some view_id) (Some my_rank) None None
  and multi_recv = array (fun rank ->
    id (Some view_id) (Some rank) None None)
  and pt2pt_send = array (fun rank ->
    id (Some view_id) (Some my_rank) (Some rank) None)
  and pt2pt_recv = array (fun rank ->
    id (Some view_id) (Some rank) (Some my_rank) None)
  and merge_send view_id dest =
    id view_id None None (Some dest)
  and merge_recv = 
    id (Some view_id) None None (Some endpt)
  and gossip =
    id None None None None 
  in

  let installed = ref [] in
  let install name kind rank c =
    (*Hashtbl.add enabled (id,c) () ;*)
    installed := (c,kind,rank,name) :: !installed
  in

  if gossip_trans then (
    install s_gossip Other (-1) gossip
  ) else (
    install s_merge Other (-1) merge_recv ;
    for rank = 0 to pred nmembers do
      if rank <> my_rank then (
	install s_send Send rank pt2pt_recv.(rank) ;
	install s_cast Cast rank multi_recv.(rank)
      )
    done
  ) ;

  {
    id		= (version,endpt,group,view_id,stack_id,proto_id) ;
    pt2pt_send 	= pt2pt_send ;
    pt2pt_recv 	= pt2pt_recv ;
    multi_send 	= multi_send ;
    multi_recv 	= multi_recv ;
    merge_send  = merge_send ;
    merge_recv  = merge_recv ;
    gossip      = gossip ;
    all_recv    = !installed
  }

(**************************************************************)
