(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* AUTH.ML *)
(* Authors: Mark Hayden, Ohad Rodeh, 8/96 *)
(**************************************************************)
(* This file gives a generic interface to authentication
 * services. *)
(**************************************************************)
open Util
open Trans

type credentials = string
type clear = string
type cipher = string

type t = {
  name   : name ;
  princ  : Addr.id -> credentials -> Addr.t ; 
  seal   : Addr.id -> Addr.set -> Addr.set -> clear -> cipher option ;
  unseal : Addr.id -> Addr.set -> Addr.set -> cipher -> clear option
}

let handlers = ref [] 

let handler_of_svc svc =
  try
    List.assoc svc !handlers 
  with Not_found -> 
    eprintf "AUTH:did not find the required service\n" ;
    raise Not_found

let create name princ seal unseal = {
  name = name ;
  princ = princ ;
  seal = seal ;
  unseal = unseal
}

let install id t = 
  handlers := (id,t) :: !handlers

let lookup id = List.assoc id !handlers

let principal t = t.princ
let seal t = t.seal
let unseal t = t.unseal

(**************************************************************)

type ticket = (Addr.t * cipher) list

let ticket src dst info =
  (* Find intersection of modes.
   *)
  let sm = Array.to_list (Addr.ids_of_set src) in
  let dm = Array.to_list (Addr.ids_of_set dst) in
  let m = Lset.intersect sm dm in
  let m = list_filter Addr.has_auth m in

  let info = 
    List.map (fun id ->
      let t = lookup id in
      let info = seal t id src dst info in
      option_map (fun info -> 
      	let addr = Addr.project src id in
      	(addr,info)
      ) info
    ) m
  in
  let info = filter_nones info in

  match info with 
  | [] -> None
  | _ ->
      Some info

let check dst info =
  let dst_ids = Addr.ids_of_set dst in
  let info = 
    List.map (fun (src,cipher) ->
      let id = Addr.id_of_addr src in
      if array_mem id dst_ids then (
      	let t = lookup id in
      	unseal t id (Addr.set_of_array [|src|]) dst cipher
      ) else None
    ) info
  in

  let info = filter_nones info in
  match info with
  | [] -> None
  | hd :: _ -> Some hd

(**************************************************************)
