(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* APPL_INTF.T: The record interface for applications.  An
 * application must define all the following callbacks and
 * put them in a record.
 *)

type (
  'cast_msg,
  'send_msg,
  'merg_msg,
  'view_msg
) full = {

  recv_cast 		: origin -> 'cast_msg ->
    ('cast_msg,'send_msg) action list ;

  recv_send 		: origin -> 'send_msg ->
    ('cast_msg,'send_msg) action list ;

  heartbeat_rate	: Time.t ;

  heartbeat 		: Time.t ->
    ('cast_msg,'send_msg) action list ;

  block 		: unit ->
    ('cast_msg,'send_msg) action list ;

  block_recv_cast 	: origin -> 'cast_msg -> unit ;
  block_recv_send 	: origin -> 'send_msg -> unit ;
  block_view            : View.full -> (rank * 'merg_msg) list ;
  block_install_view    : View.full -> 'merg_msg list -> 'view_msg ;
  unblock_view 		: View.full -> 'view_msg ->
    ('cast_msg,'send_msg) action list ;

  exit 			: unit -> unit
}

(**************************************************************)

type (
  'cast_msg,
  'send_msg,
  'merg_msg,
  'view_msg
) header =
  | TCast of 'cast_msg
  | TSend of 'send_msg
  | TMerge of (rank,'merge_msg) list
  | TView of 'view_msg

type 'merge_msg state = 
  | Xfer('merge_msg option array)
  | Normal
  | Blocked

(**************************************************************)

let al al =
  List.iter (function
    | Cast(m) -> Cast(TCast(m)
    | Send(dests,m) -> Send(dests,TSend(m))
    | Leave -> Leave
    | XferDone -> XferDone
    | Dump -> Dump
    | Protocol p -> Protocol p
    | _ -> ()
  ) al ;
  al

let simpleXfer intf =
  let state = ref Xfer in
  let buffer = Queue.create () in

  let rec recv from msg =
    match !state,msg with
    | Normal,TCast(msg) -> al (intf.recv_cast from msg)
    | Normal,TSend(msg) -> al (intf.recv_send from msg)
    | Blocked,TCast(msg) -> al (intf.block_recv_cast from msg)
    | Blocked,TSend(msg) -> al (intf.block_recv_send from msg)
    | Xfer(_),TCast(_) -> Queue.add buffer (from,msg)
    | Xfer(_),TSend(_) -> Queue.add buffer (from,msg)
    | Xfer(states),Merge(merge) ->
	recv_merge merge
    | Xfer(_),View(view) ->
	recv_view view

  and recv_view view =
    state := Normal ;
    let a1 = al (intf.unblock_view (ls,vs) msg) in
    let buf = list_of_queue buffer in
    Queue.clear buffer ;
    let a2 =
      List.map (function (rank,msg) ->
      	match msg with
      	| TCast(msg) -> al (intf.recv_cast rank msg)
      	| TSend(msg) -> al (intf.recv_send rank msg)
      ) buf
    in
    a1 @ a2

  and check_ready () =
    match state with
    | Xfer(states) ->
    	if not (array_mem None states) then (
	  state := Normal ;
	  
	  (* Put all of the state info into a list.
	   *)
      	  let states = Array.map some_of s.states in
	  let states = Array.to_list states in
      	  let view = s.intf.block_install_view (!ls,!vs) states in
	  let a1 = [Cast(View(view_msg))] in
	  let a2 = recv_view view in
	  a1 @ a2
        ) else []
    | _ -> []

  and recv_merge merge =
    List.iter (fun (rank,merge) ->
      states.(rank) <- Some merge
    ) merge ; 
    check_ready ()
  in

  and heartbeat time msg =
    match !state with
    | Normal -> al (intf.heartbeat time)
    | _ -> []
  and block () =
    state := Idle ;
    al (intf.block ())
  and install_view (ls,vs) =
    ls := ls ;
    vs := vs ;
    state := Xfer(array_create name ls.nmembers None) ;
    let merge = block_view (ls,vs) in
    if ls.am_coord then (
      recv_merge merge
    ) else (
      [Send(vs.coord,(Merge(merge)))]
    )
  and exit () = intf.exit ()

  in {
    recv_cast           = recv_cast ;
    recv_send           = recv_send ;
    block               = block ;
    heartbeat           = heartbeat ;
    heartbeat_rate      = intf.heartbeat_rate ;
    block_recv_cast     = block_recv_cast ;
    block_recv_send     = block_recv_send ;
    block_view          = block_view ;
    block_install_view  = block_install_view ;
    unblock_view        = unblock_view ;
    exit                = exit
  }

(**************************************************************)

