(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* APPL_INTF.ML: application interface *)
(* Author: Mark Hayden, 8/95 *)
(* See documentation for a description of this interface *)
(**************************************************************)
open Util
(**************************************************************)

(* Some type aliases.
 *)
type rank	= int
type view 	= Endpt.id list
type mergers 	= Endpt.id list
type contact 	= Endpt.id
type origin 	= rank
type dests 	= rank list

(**************************************************************)
(* ACTION: The type of actions an application can take.
 * These are typically returned from callbacks as lists of
 * actions to be taken.

 * Cast(msg): Broadcast a message to entire group

 * Send(dests,msg): Send a message to subset of group.

 * Leave: Leave the group.

 * XferDone: Mark end of a state transfer. (Not supported by
 * all protocol stacks.)

 * Protocol(protocol): Request a protocol switch.  Will
 * cause a view change to new protocol stack.  (Not
 * supported by all protocol stacks.)

 * Migrate(addresses): Change address list for this member.
 * Will cause a view change.

 * Timeout(time): Request a timeout at a particular time.
 * The timeout takes the form of a heartbeat callback.

 * Dump: Undefined (for debugging purposes)
 *)

type ('cast_msg, 'send_msg) action =
  | Cast of 'cast_msg
  | Send of dests * 'send_msg
  | Leave
  | XferDone
  | Protocol of Proto.id
  | Migrate of Addr.set
  | Rekey
  | Suspect of rank list
  | Prompt
  | Timeout of Time.t
  | Dump
  | Block of bool			(* not for casual use *)

(**************************************************************)
(* APPL_INTF.FULL: The record interface for applications.  An
 * application must define all the following callbacks and
 * put them in a record.
 *)

type (
  'cast_msg,
  'send_msg,
  'merg_msg,
  'view_msg
) full = {

  recv_cast 		: origin -> 'cast_msg ->
    ('cast_msg,'send_msg) action list ;

  recv_send 		: origin -> 'send_msg ->
    ('cast_msg,'send_msg) action list ;

  heartbeat_rate	: Time.t ;

  heartbeat 		: Time.t ->
    ('cast_msg,'send_msg) action list ;

  block 		: unit ->
    ('cast_msg,'send_msg) action list ;

  block_recv_cast 	: origin -> 'cast_msg -> unit ;
  block_recv_send 	: origin -> 'send_msg -> unit ;
  block_view            : View.full -> (rank * 'merg_msg) list ;
  block_install_view    : View.full -> 'merg_msg list -> 'view_msg ;
  unblock_view 		: View.full -> 'view_msg ->
    ('cast_msg,'send_msg) action list ;

  exit 			: unit -> unit
}

(**************************************************************)

(* A debugging interface: it takes an interface as an argument
 * and wraps itself around the interface, printing debugging
 * information on most events.
 *)
val debug : ('a,'b,'c,'d) full -> ('a,'b,'c,'d) full

(**************************************************************)

type iov = (Iovec.t, Iovec.t, Iovec.t, Iovec.t) full

type iovl = (Iovecl.t, Iovecl.t, Iovecl.t, Iovecl.t) full

type ('a,'b,'c,'d) power = ('a * Iovecl.t, 'b * Iovecl.t,'c,'d) full

type t = iovl

(**************************************************************)
(* Conversion functions for the various types
 * of interfaces.
 *)

val full : ('a,'b,'c,'d) full -> t
val aggr : ('a,'b,'c,'d) full -> t	(* aggregates messages *)

val iov : iov -> t

val iovl : iovl -> t

val power : ('a,'b,'c,'d) power -> t

(**************************************************************)
