(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* APPL_INTF.ML: application interface *)
(* Author: Mark Hayden, 8/95 *)
(* See documentation for a description of this interface *)
(**************************************************************)
open Util
(**************************************************************)
let name = Trace.source_file "APPL_INTF"
let failwith s = failwith (Util.failmsg name s)
let log = Trace.log name ""
(**************************************************************)

(* Some type aliases.
 *)
type rank	= int
type view 	= Endpt.id list
type mergers 	= Endpt.id list
type contact 	= Endpt.id
type origin 	= rank
type dests 	= rank list

(**************************************************************)

type ('cast_msg, 'send_msg) action =
  | Cast of 'cast_msg
  | Send of dests * 'send_msg
  | Leave
  | XferDone
  | Protocol of Proto.id
  | Migrate of Addr.set
  | Rekey
  | Suspect of rank list
  | Prompt
  | Timeout of Time.t
  | Dump
  | Block of bool			(* not for casual use *)

(**************************************************************)
(* APPL_INTF.T: The record interface for applications.  An
 * application must define all the following callbacks and
 * put them in a record.
 *)

type (
  'cast_msg,
  'send_msg,
  'merg_msg,
  'view_msg
) full = {

  recv_cast 		: origin -> 'cast_msg ->
    ('cast_msg,'send_msg) action list ;

  recv_send 		: origin -> 'send_msg ->
    ('cast_msg,'send_msg) action list ;

  heartbeat_rate	: Time.t ;

  heartbeat 		: Time.t ->
    ('cast_msg,'send_msg) action list ;

  block 		: unit ->
    ('cast_msg,'send_msg) action list ;

  block_recv_cast 	: origin -> 'cast_msg -> unit ;
  block_recv_send 	: origin -> 'send_msg -> unit ;
  block_view            : View.full -> (rank * 'merg_msg) list ;
  block_install_view    : View.full -> 'merg_msg list -> 'view_msg ;
  unblock_view 		: View.full -> 'view_msg ->
    ('cast_msg,'send_msg) action list ;

  exit 			: unit -> unit
}

(**************************************************************)

let debug_msg name =
  eprintf "APPL:%s\n" name

let al al =
  List.iter (function
    | Cast(_) -> 
	debug_msg "CAST"
    | Send(dests,_) ->
	debug_msg "SEND"
    | Leave ->
	debug_msg "LEAVE"
    | XferDone ->
	debug_msg "XFER_DONE"
    | Dump ->
	debug_msg "DUMP"
    | Protocol p ->
	let msg = sprintf "PROTOCOL:%s" (Proto.string_of_id p) in
	debug_msg msg
    | Prompt -> 
	debug_msg "PROMPT"
    | _ -> ()
  ) al ;
  al

let debug intf =
  let recv_cast from msg =
    debug_msg "RECV_CAST" ;
    al (intf.recv_cast from msg)
  and recv_send from msg =
    debug_msg "RECV_SEND" ;
    al (intf.recv_send from msg)
  and block () =
    debug_msg "BLOCK" ;
    al (intf.block ())
  and heartbeat time =
    debug_msg (sprintf "HEARTBEAT(%s)" (Time.to_string time)) ;
    al (intf.heartbeat time)
  and block_recv_cast from msg =
    debug_msg "BLOCK_RECV_CAST" ;
    intf.block_recv_cast from msg
  and block_recv_send from msg =
    debug_msg "BLOCK_RECV_SEND" ;
    intf.block_recv_send from msg
  and block_view contact =
    debug_msg "BLOCK_VIEW" ;
    intf.block_view contact
  and block_install_view view =
    debug_msg "BLOCK_INSTALL_VIEW" ;
    intf.block_install_view view
  and unblock_view (ls,vs) msg =
    debug_msg "UNBLOCK_VIEW" ;
(*
    eprintf "APPL:(view=%s)\n" (Endpt.string_of_id_list vs.View.view) ;
*)
    eprintf "APPL:%s\n" (View.string_of_full (ls,vs)) ;
    al (intf.unblock_view (ls,vs) msg)
  and exit () =
    debug_msg "EXIT" ;
    intf.exit ()
  in {
    recv_cast           = recv_cast ;
    recv_send           = recv_send ;
    block               = block ;
    heartbeat           = heartbeat ;
    heartbeat_rate      = intf.heartbeat_rate ;
    block_recv_cast     = block_recv_cast ;
    block_recv_send     = block_recv_send ;
    block_view          = block_view ;
    block_install_view  = block_install_view ;
    unblock_view        = unblock_view ;
    exit                = exit
  }

(**************************************************************)
(**************************************************************)

let make_marsh name =
  let name = "APPL_INTF:"^name in
  Iovecl.make_marsh name

(**************************************************************)
exception Slow

let power_marsh name = 
  let name = "APPL_INTF(pow):"^name in
  let scribble = String.create 4 in
  let imarsh,iunmarsh = Mbuf.make_marsh name Mbuf.global in
  let marsh (obj,iovl) = 
    let obj_iov = imarsh obj in
    Hsys.push_int scribble 0 (Iovec.len name obj_iov) ;
    let scribble = String.copy scribble in
    let rbuf = Iovec.heap name scribble in
    let len_iov = Iovec.alloc name rbuf 0 (String.length scribble) in
    Array.append [|len_iov;obj_iov|] iovl
  and unmarsh iovl =
    let iov = Iovecl.flatten name iovl in
    Iovec.read name iov (fun buf ofs len ->
      let obj_len = Hsys.pop_int buf ofs in
      let obj_iov = Iovec.sub name iov (ofs+4) obj_len in
      let obj = Iovec.read name obj_iov iunmarsh in
      let rest = Iovec.sub name iov (ofs+4+obj_len) (len-4-obj_len) in
      let rest = [|rest|] in
      Iovec.free name iov ;
      (obj,rest)
    )
  in
  
  let unmarsh iovl =
    try
      let niov = Array.length iovl in
      if niov <| 1 then (
      	log (fun () -> sprintf "#iovecs=%d < 1" niov) ;
      	raise Slow
      ) ;
      
      let len0, obj_len = 
      	Iovec.read name iovl.(0) (fun buf ofs len ->
 	  if len <| 4 then (
	    log (fun () -> sprintf "len=%d < 4" len) ;
	    raise Slow ;
	  ) ;
	  len, (Hsys.pop_int buf ofs)
        )
      in

      if len0 =| 4 && niov >=| 2 then (
      	Iovec.read name iovl.(1) (fun buf ofs len ->
	  if len <| obj_len then
	    raise Slow ;
	  let obj_iov = Iovec.sub name iovl.(1) ofs obj_len in
	  let obj = Iovec.read name obj_iov iunmarsh in
	  let iovl =
	    if obj_len =| len then (
	      Array.sub iovl 2 (niov - 2)
	    ) else (
	      let iov1 = Iovec.sub name iovl.(1) (ofs+obj_len) (len-obj_len) in
	      Iovec.free name iovl.(0) ;
	      Iovec.free name iovl.(1) ;
	      let iovl = Array.sub iovl 1 (niov - 1) in (* Be conservative here. *)
	      iovl.(0) <- iov1 ;
	      iovl
	    )
          in 
	  Iovec.free name obj_iov ;
	  (obj,iovl)
        )
      ) else (
      	Iovec.read name iovl.(0) (fun buf ofs len ->
	  if len <| obj_len + 4 then
	    raise Slow ;
	  let obj_iov = Iovec.sub name iovl.(0) (ofs+4) obj_len in
	  let obj = Iovec.read name obj_iov iunmarsh in
	  let iov0 = Iovec.sub name iovl.(0) (ofs+4+obj_len) (len-4-obj_len) in
	  Iovec.free name iovl.(0) ;
	  let iovl = Array.copy iovl in (* Be conservative here. *)
	  iovl.(0) <- iov0 ;
	  Iovec.free name obj_iov ;
	  (obj,iovl)
	)
      )
    with Slow ->
      unmarsh iovl
  in
 
  (marsh,unmarsh)
(*
let power_marsh name = 
  let ma,um = make_marsh ("APPL_INTF:"^name) in
  let ma (obj,iovl) = 
    let iovl = Iovecl.flatten name iovl in
    let iovl = Iovec.to_string name iovl in
    ma (obj,iovl)
  and um iovl =
    let (obj,iovl) = um iovl in
    let iovl = Iovec.of_string name iovl in
    let iovl = [|iovl|] in
    (obj,iovl)
  in (ma,um)
*)
(**************************************************************)

let full i =
  let cast_ma,cast_um = make_marsh "cast" in
  let send_ma,send_um = make_marsh "send" in
  let merge_ma,merge_um = make_marsh "merge" in
  let view_ma,view_um = make_marsh "view" in
  let merge_ma1 = List.map (fun (rank,m) -> (rank,merge_ma m)) in
  let merge_um1 = List.map merge_um in

  let ta = 
    List.map (function
    | Cast m -> Cast(cast_ma m)
    | Send(d,m) -> Send(d,(send_ma m))
    | Leave -> Leave
    | XferDone -> XferDone
    | Protocol a -> Protocol a
    | Timeout a -> Timeout a
    | Migrate a -> Migrate a
    | Dump -> Dump
    | Rekey -> Rekey
    | Suspect a -> Suspect a
    | Prompt -> Prompt
    | Block b -> Block b)
  in {
    recv_cast = (fun o m -> ta (i.recv_cast o (cast_um m))) ;
    recv_send = (fun o m -> ta (i.recv_send o (send_um m))) ;
    heartbeat = (fun t -> ta (i.heartbeat t)) ;
    heartbeat_rate = i.heartbeat_rate ;
    block = (fun () -> ta (i.block ())) ;
    block_recv_cast = (fun o m -> i.block_recv_cast o (cast_um m)) ;
    block_recv_send = (fun o m -> i.block_recv_send o (send_um m)) ;
    block_view = (fun v -> merge_ma1 (i.block_view v)) ;
    block_install_view = (fun v s -> view_ma (i.block_install_view v (merge_um1 s))) ;
    unblock_view = (fun vs m -> ta (i.unblock_view vs (view_um m))) ;
    exit = i.exit
  }

(**************************************************************)

let aggr i =
  let cast_ma,cast_um = make_marsh "cast" in
  let send_ma,send_um = make_marsh "send" in
  let merge_ma,merge_um = make_marsh "merge" in
  let view_ma,view_um = make_marsh "view" in
  let merge_ma1 = List.map (fun (rank,m) -> (rank,merge_ma m)) in
  let merge_um1 = List.map merge_um in

  let pending = ref [] in
  let nmembers = ref 0 in
  let next_sweep = ref Time.zero in

  let collect actions =
    pending := actions :: !pending
  in    

  let emit () =
    let casts = ref [] in
    let sends = array_create name !nmembers [] in
    let other = ref [] in

    (* Go through the outer list in order, and the
     * inner list in reverse order.
     *)
    List.iter (fun al -> 
      List.iter (fun a -> match a with
      | Cast(m) -> 
	  casts := m :: !casts
      | Send(dl,m) -> 
	  List.iter (fun d -> sends.(d) <- m :: sends.(d)) dl
      | Leave -> other := Leave :: !other
      | XferDone -> other := XferDone :: !other
      | Protocol a -> other := Protocol a :: !other
      | Timeout a -> other := Timeout a :: !other
      | Migrate a -> other := Migrate a :: !other
      | Dump -> other := Dump :: !other
      |	Suspect a -> other := Suspect a :: !other
      | Rekey -> other := Rekey :: !other
      | Prompt -> other := Prompt :: !other
      | Block a -> other := Block a :: !other
      ) (List.rev al)
    ) !pending ;
    pending := [] ;

    let emit = other in
    
    for i = 0 to pred !nmembers do
      if sends.(i) <> [] then
	emit := Send([i], send_ma sends.(i)) :: !emit
    done ;
    if !casts <> [] then
      emit := Cast(cast_ma !casts) :: !emit ;
    !emit
  in

  let heartbeat time =
    if time > !next_sweep then (
      next_sweep := Time.add time i.heartbeat_rate ;
      emit ()
    ) else []
  in

  let block () =
    emit ()
  in

  let unblock_view (ls,vs) =
    nmembers := ls.View.nmembers ;
    emit ()
  in
  
  { recv_cast = (fun o m -> List.iter (fun m -> collect (i.recv_cast o m)) (cast_um m) ; []) ;
    recv_send = (fun o m -> List.iter (fun m -> collect (i.recv_send o m)) (send_um m) ; []) ;
    heartbeat = (fun t -> collect (i.heartbeat t) ; heartbeat t) ;
    heartbeat_rate = i.heartbeat_rate ;
    block = (fun () -> collect (i.block ()) ; block ()) ;
    block_recv_cast = (fun o m -> List.iter (fun m -> i.block_recv_cast o m) (cast_um m)) ;
    block_recv_send = (fun o m -> List.iter (fun m -> i.block_recv_send o m) (send_um m)) ;
    block_view = (fun v -> merge_ma1 (i.block_view v)) ;
    block_install_view = (fun v s -> view_ma (i.block_install_view v (merge_um1 s))) ;
    unblock_view = (fun vs m -> collect (i.unblock_view vs (view_um m)); unblock_view vs) ;
    exit = i.exit
  }

(**************************************************************)

let power i =
  let cast_ma,cast_um = power_marsh "cast" in
  let send_ma,send_um = power_marsh "send" in
  let merge_ma,merge_um = make_marsh "merge" in
  let view_ma,view_um = make_marsh "view" in
  let merge_ma1 = List.map (fun (rank,m) -> (rank,merge_ma m)) in
  let merge_um1 = List.map merge_um in

  let ta = 
    List.map (function
    | Cast m -> Cast(cast_ma m)
    | Send(d,m) -> Send(d,(send_ma m))
    | Leave -> Leave
    | XferDone -> XferDone
    | Protocol a -> Protocol a
    | Migrate a -> Migrate a
    | Timeout a -> Timeout a
    | Suspect a -> Suspect a
    | Rekey -> Rekey
    | Dump -> Dump
    | Prompt -> Prompt
    | Block b -> Block b)
  in {
    recv_cast = (fun o m -> ta (i.recv_cast o (cast_um m))) ;
    recv_send = (fun o m -> ta (i.recv_send o (send_um m))) ;
    heartbeat = (fun t -> ta (i.heartbeat t)) ;
    heartbeat_rate = i.heartbeat_rate ;
    block = (fun () -> ta (i.block ())) ;
    block_recv_cast = (fun o m -> i.block_recv_cast o (cast_um m)) ;
    block_recv_send = (fun o m -> i.block_recv_send o (send_um m)) ;
    block_view = (fun v -> merge_ma1 (i.block_view v)) ;
    block_install_view = (fun v s -> view_ma (i.block_install_view v (merge_um1 s))) ;
    unblock_view = (fun vs m -> ta (i.unblock_view vs (view_um m))) ;
    exit = i.exit
  }

(**************************************************************)

let iov i =
  let flat = Iovecl.flatten name in
  let ta = 
    List.map (function
    | Cast m -> Cast([|m|])
    | Send(d,m) -> Send(d,[|m|])
    | Leave -> Leave
    | XferDone -> XferDone
    | Protocol a -> Protocol a
    | Migrate a -> Migrate a
    | Suspect a -> Suspect a
    | Timeout a -> Timeout a
    | Rekey -> Rekey
    | Dump -> Dump
    | Prompt -> Prompt
    | Block b -> Block b)
  in {
    recv_cast = (fun o m -> ta (i.recv_cast o (flat m))) ;
    recv_send = (fun o m -> ta (i.recv_send o (flat m))) ;
    heartbeat = (fun t -> ta (i.heartbeat t)) ;
    heartbeat_rate = i.heartbeat_rate ;
    block = (fun () -> ta (i.block ())) ;
    block_recv_cast = (fun o m -> i.block_recv_cast o (flat m)) ;
    block_recv_send = (fun o m -> i.block_recv_send o (flat m)) ;
    block_view = (fun v -> List.map (fun (a,b) -> (a,[|b|])) (i.block_view v)) ;
    block_install_view = (fun v s -> 
      let s = List.map flat s in
      [|i.block_install_view v s|]) ;
    unblock_view = (fun vs m -> ta (i.unblock_view vs (flat m))) ;
    exit = i.exit
  }

(**************************************************************)

let iovl a = a

(**************************************************************)

type iov = (Iovec.t, Iovec.t, Iovec.t, Iovec.t) full

type iovl = (Iovecl.t, Iovecl.t, Iovecl.t, Iovecl.t) full

type ('a,'b,'c,'d) power = ('a * Iovecl.t, 'b * Iovecl.t,'c,'d) full

(**************************************************************)

type t = iovl

(**************************************************************)
