(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* ALARM.ML *)
(* Author: Mark Hayden, 4/96 *)
(**************************************************************)
open Util
open Trans
open Hsys
(**************************************************************)
let name = Trace.source_file "ALARM"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type id =
  | Netsim
  | Real
  | Tk
  | Threaded

type alarm = {
  disable : unit -> unit ;
  schedule : Time.t -> unit
}

type poll_type = SocksPolls | OnlyPolls

type t = {
  name		: string ;		(* name *)
  gettime	: unit -> Time.t ;	(* get current time *)
  alarm		: (Time.t -> unit) -> alarm ;
  check		: unit -> bool ;	(* check the timers *)
  min		: unit -> Time.t ;	(* get next timer *)
  add_sock	: socket -> handler -> unit ;
  rmv_sock	: socket -> unit ;
  block		: unit -> unit ;	(* block until next alarm *)
  add_poll	: string -> (bool -> bool) -> unit ;
  rmv_poll	: string -> unit ;
  poll		: poll_type -> bool
}

(**************************************************************)

let mapping = [
  "NETSIM",   Netsim ;
  "REAL",     Real ;
  "TK",       Tk ;
  "THREADED", Threaded
] 

let id_of_string,string_of_id = make_map name "id" mapping

(**************************************************************)

let c_alarm dis sch = {
  disable = dis ;
  schedule = sch
}

let disable a = a.disable ()
let schedule a = a.schedule

let create
  name
  gettime
  alarm
  check
  min
  add_sock
  rmv_sock
  block
  add_poll
  rmv_poll
  poll
= {
    name	= name ;
    gettime     = gettime ;
    check       = check ;
    alarm       = alarm ;
    min         = min ;
    add_sock	= add_sock ;
    rmv_sock	= rmv_sock ;
    add_poll	= add_poll ;
    rmv_poll	= rmv_poll ;
    poll	= poll ;
    block	= block
  }

(**************************************************************)

let gettime t = t.gettime ()
let alarm t = t.alarm
let check t = t.check ()
let min t = t.min ()
let add_sock t = t.add_sock
let rmv_sock t = t.rmv_sock
let block t = t.block ()
let add_poll t = t.add_poll
let rmv_poll t = t.rmv_poll
let poll t = t.poll
let name t = t.name

(**************************************************************)

let poll_f 		= ref Util.ident

let update_polls polls =
  match Resource.to_list2 polls with
  | [] -> poll_f := Util.ident
  | [p1] -> 
      poll_f := p1
  | [p1;p2] -> 
      poll_f := (fun b -> p2 (p1 b))
  | [p1;p2;p3] -> 
      poll_f := (fun b -> p3 (p2 (p1 b)))
  | pl ->
      let rec loop rm = function
        | poll::tl -> loop (poll rm) tl
        | [] -> rm
      in 
      poll_f := (fun b -> loop b pl)

let polls =
  Resource.create "ALARM:polls" 
  (fun _ _ -> ())
  (fun _ -> ())
  update_polls

let alm_add_poll (n:string) (poll:bool->bool) = 
  Resource.add polls n poll

let alm_rmv_poll n =
  Resource.remove polls n

let alm_poll b = !poll_f b

(**************************************************************)

(* For bypass stuff.
 *)
let wrap wrapper alarm =
  let enable callback =
    alarm.alarm (wrapper callback)
  in {
    name	= alarm.name ;
    alarm 	= enable ;
    gettime	= alarm.gettime ;
    min		= alarm.min ;
    check	= alarm.check ;
    block	= alarm.block ;
    add_sock	= alarm.add_sock ;
    rmv_sock	= alarm.rmv_sock ;
    add_poll	= alarm.add_poll ;
    rmv_poll 	= alarm.rmv_poll ;
    poll	= alarm.poll
  }

(**************************************************************)
(* Alarm management.
 *)

let alarms = ref []

let install id a =
  alarms := (id,a) :: !alarms

let alarm_of_id id = 
  try List.assoc id !alarms with Not_found -> (
    eprintf "ARGE:alarm not installed, exiting" ;
    exit 1
  )

let chosen = ref None

let choose id sched =
  let a = alarm_of_id id in
  if !chosen = None then (
    chosen := Some(a sched)
  )

let get () = 
  match !chosen with
  | None -> failwith "no alarm chosen"
  | Some a -> a

(**************************************************************)
