(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* ADDR.ML *)
(* Author: Mark Hayden, 12/95 *)
(**************************************************************)
let name = Trace.source_file "ADDR"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)
open Util
open Trans
(**************************************************************)

type id =
  | Atm
  | Deering
  | Netsim
  | Tcp
  | Udp
  | Sp2
  | Krb5
  | Pgp
  | Fortezza

let all = [|Atm;Deering;Netsim;Tcp;Udp;Sp2;Krb5;Pgp;Fortezza|]

type t =
  | AtmA of inet
  | DeeringA of inet * port
  | NetsimA
  | TcpA of inet * port
  | UdpA of inet * port
  | Sp2A of inet * port
  | Krb5A of string
  | PgpA of string
  | FortezzaA of string

type set = t array

let id_of_addr = function
  | NetsimA    -> Netsim
  | TcpA     _ -> Tcp
  | UdpA     _ -> Udp
  | DeeringA _ -> Deering
  | AtmA     _ -> Atm
  | Sp2A     _ -> Sp2
  | Krb5A    _ -> Krb5
  | PgpA     _ -> Pgp
  | FortezzaA _ -> Fortezza

let ids_of_set = Array.map id_of_addr

let mapping = [
  "ATM", Atm ;
  "DEERING", Deering ;
  "NETSIM", Netsim ;
  "TCP", Tcp ;
  "UDP", Udp ;
  "SP2", Sp2 ;
  "PGP", Pgp ;
  "FORTEZZA", Fortezza ;
  "KRB5", Krb5
] 

let id_of_string,string_of_id = make_map name "id" mapping

let set_of_array = ident
let array_of_set = ident

let string_of_id_short id =
  let id = string_of_id id in
  try String.sub id 0 1 with
  | _ -> failwith "string_of_id_short:sanity"

let string_of_inet_port = 
  string_of_pair Hsys.string_of_inet string_of_int

let string_of_addr = function
  | NetsimA    -> "Netsim"
  | TcpA     (i,p) -> sprintf "Tcp%s" (string_of_inet_port (i,p))
  | UdpA     (i,p) -> sprintf "Udp%s" (string_of_inet_port (i,p))
  | DeeringA (i,p) -> sprintf "Deering%s" (string_of_inet_port (i,p))
  | AtmA     i     -> sprintf "Atm(%s)" (Hsys.string_of_inet i)
  | Sp2A     (i,p) -> sprintf "Sp2%s" (string_of_inet_port (i,p))
  | PgpA         _ -> sprintf "Pgp(_)"
  | Krb5A        _ -> sprintf "Krb5(_)"
  | FortezzaA    _ -> sprintf "Fortezza(_)"

let string_of_set = string_of_array string_of_addr

let has_mcast = function
  | Udp
  | Deering 
  | Netsim -> true
  | _ -> false

let has_pt2pt = function
  | Udp
  | Deering
  | Tcp
  | Atm
  | Netsim 
  | Sp2 -> true
  | _ -> false

let has_auth = function
  | Pgp
  | Krb5
  | Fortezza -> true
  | _ -> false
      
let all = [Atm;Deering;Sp2;Udp;Tcp;Netsim]

let prefer =
  let ranking = function
  | Atm -> 5
  | Deering -> 4
  | Sp2 -> 3
  | Udp -> 2
  | Tcp -> 1
  | Netsim -> 0
  | _ -> -1
  in

  let init = ((-1),None) in

  fun modes ->
    match 
      array_fold_left (fun ((mx,_) as it) m ->
	let rank = ranking m in
	if rank >= 0 && rank > mx then (rank,Some m) else it
      ) init modes
    with
    | _,None ->
      	eprintf "ADDR:modes=%s\n" (string_of_array string_of_id modes) ;
      	failwith "no such transport available"
    | _,(Some mode) -> mode
(*
    printf "ADDR:modes=%s,cast=%b,res=%s\n"
      (string_of_list string_of_id modes) cast 
      (string_of_id (List.nth preferences rank)) ;
*)

let project addrs mode =
  let rec loop i =
    if i >= Array.length addrs then
      failwith "project:mode not available"      
    else if id_of_addr addrs.(i) = mode then
      addrs.(i)
    else loop (succ i)
  in loop 0

let check a =
  let m = id_of_addr a in
  has_pt2pt m || has_mcast m

let same_process a b =
  let a = array_filter check a in
  let b = array_filter check b in
  let a = Array.to_list a in
  let b = Array.to_list b in
  not (Lset.disjoint a b)

let compress me addrs =
  let local,addrs = 
    array_fold_left (fun (local,remote) hd ->
      (* 1. If on same host, mark local and strip.
       * 2. If dest already included, strip.
       * 3. Otherwise add to list
       *)
      if same_process hd me then
	(true,remote)
      else if List.exists (same_process hd) remote then
	(local,remote)
      else
	(local, (hd :: remote))
    ) (false,[]) addrs
  in
  let addrs = Array.of_list addrs in
  (local,addrs)
(*
let compress me o =
  let r = compress me o in
  eprintf "ADDR:compress:me=%s" (string_of_set me) ;
  eprintf "  input =%s\n" (string_of_list string_of_set o) ;
  eprintf "  output=%s,%b\n" 
    (string_of_list string_of_set (snd r)) (fst r) ;
  r
*)
(**************************************************************)

let modes_of_view addrs =
  let modes = Array.map (Array.map id_of_addr) addrs in
  let modes = Array.map Array.to_list modes in
  let modes = array_fold_left Lset.intersect all modes in
  if modes = [] then (
    eprintf "ADDR:modes_of_view:warning:empty:%s\n"
      (string_of_array string_of_set addrs)
  ) ;
  Array.of_list modes

(**************************************************************)

let error modes =
  let modes = Array.to_list modes in
  let soms m = string_of_list string_of_id m in
  eprintf "

Error.  A group is being initialized with a bad set of
communication modes.

  Selected modes were: %s
  Of which, these modes support pt2pt: %s
  And these modes support mcast: %s

Each member of a group must have at least one mcast and one
pt2pt mode ennabled.  In addition, all members must share at
least one common mode of both pt2pt and mcast type in order
to communicate.

  All Ensemble modes are: %s
  Of which, these have pt2pt: %s
  And these support mcast: %s

Note that not all modes may be available to every
application (for example, DEERING requires IP multicast).

Modes are selected through the ENS_MODES environment
variable and the -modes command-line argument, both of which
give a colon (':') separated list of modes to use (for
example, DEERING:UDP).

(exiting)
"
    (soms modes)
    (soms (list_filter has_pt2pt modes))
    (soms (list_filter has_mcast modes))

    (soms all)
    (soms (list_filter has_pt2pt all))
    (soms (list_filter has_mcast all)) ;
  exit 1

(**************************************************************)
