(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* UDP.ML *)
(* Author: Mark Hayden, 5/95 *)
(**************************************************************)
open Util
open Trans
(**************************************************************)
let name = Trace.source_file "UDP"
let failwith s = failwith (Util.failmsg name s)
let log = Trace.log name ""
(**************************************************************)
(* NOTE: What about using an extra socket set to
 * no-local-receipt for sending multicasts on?  *)
(**************************************************************)
let ts_xmit = Timestamp.add "UDP:xmit"
(**************************************************************)

let init host port =
  let sock = Hsys.socket_dgram () in
  begin try 
    (* This buffer size was suggested by Werner Vogels.
     *)
    let size = 64*1024*1024/(2*128+1024) (* = 52428 *) in

    (* Try to set the socket up to be nonblocking and
     * have been send/receive buffers.
     *)
    Hsys.setsockopt sock (Hsys.Nonblock) ;
    Hsys.setsockopt sock (Hsys.Sendbuf size) ;
    Hsys.setsockopt sock (Hsys.Recvbuf size) ;
  with e ->
    log (fun () -> sprintf "error:setsockopt:%s" (Hsys.error e))
  end ;

  (* Set the socket to close on exec.
   *)
  (*set_close_on_exec sock ;*)

  let rec loop port =
    try
      Arge.check_port name port ;
      Hsys.bind sock host port ;
      port
    with e ->
      log (fun () -> Hsys.error e) ;
      loop (succ port)
  in

  let port = loop port in

  (sock,port)

(**************************************************************)

let domain alarm =
  (* Note that deering port is always decided by the
   * configuration (whereas the general UDP port may be
   * different if the default is already in use).  We should
   * be able to bind to this port without problems because
   * all Ensemble processes use reusable IP multicast
   * sockets.  
   *)
  let deering_port = Arge.check name Arge.port in
  let gossip_port  = Arge.check name Arge.gossip_port in
  let gossip_hosts = Array.of_list (Arge.check name Arge.gossip_hosts) in

  (* This allows the user to override the host name to use to
   * send messages to me.
   *)
  let host =
    match Arge.get Arge.udp_host with
    | None -> 
  	let host = Hsys.gethostname () in
  	let host = Hsys.inet_of_string host in
	host
    | Some host -> host
  in

  let sp2_info =
    let info = ref None in
    fun () ->
      match !info with
      |	Some info -> info
      |	None ->
	  let host = Hsys.gethostname () in
	  let try_suffix suffix =
	    try
	      match string_split "." host with
	      | [p1;p2;p3;p4] ->
		  let p1 = p1 ^ suffix in
		  let host = [p1;p2;p3;p4] in
		  let host = String.concat "." host in
		  eprintf "UDP:trying sp2 udp hostname:'%s'\n" host ;
		  let host = Hsys.inet_of_string host in
		  Some(host)
	      | _ -> failwith "bad inet list"
	    with _ ->
	      None
	  in
	  let suffixes = Arge.get Arge.sp2_suffixes in
	  let rec loop = function
	    | [] -> failwith "no suffixes worked for sp2 fast interconnect hostname"
	    | hd :: tl ->
		match try_suffix hd with
		| None -> loop tl
		| Some inet ->
		    let (sock,port) = init inet deering_port in
		    let addr = Addr.Sp2A(inet,port) in
		    info := Some (sock,addr) ;
		    (sock,addr)
	  in loop suffixes
  in

  let (sock, port) = init host deering_port in
  Unique.install_port port ;

  (* TODO: I'm having problems with the loopback stuff.
   *)
  let noloop = sock in
(*
  let noloop = socket_dgram () in
  setsock_multicast noloop false ;
*)

  let udp_addr = Addr.UdpA(host,port) in
  let deering_addr = Addr.DeeringA(host,port) in

  let addr mode = match mode with
  | Addr.Udp     -> udp_addr
  | Addr.Deering -> deering_addr
  | Addr.Sp2     -> snd (sp2_info ())
  | _ -> failwith "addr:bad mode"
  in

  let xmit mode dest =
    (* Do some preprocessing.
     *)
    let dests =
      match dest with
      | Domain.Pt2pt dests ->
	  if Array.length dests = 0 then 
	    None
	  else (
	    (* Depending on the type of the address, use
	     * either the standard Udp socket or the
	     * special Sp2 socket.  
	     *)
	    let sock = match mode with
	    | Addr.Udp -> sock
	    | Addr.Deering -> sock
	    | Addr.Sp2 -> fst (sp2_info ())
	    | _ -> failwith "xmit:sanity"
	    in

	    let dests =
	      Array.map (fun dest ->
		match Addr.project dest mode with 
		| Addr.UdpA    (inet,port) -> (inet,port)
		| Addr.DeeringA(inet,port) -> (inet,port)
		| Addr.Sp2A    (inet,port) -> (inet,port)
		| _ -> failwith "xmit:sanity"
	      ) dests
	    in
	    Some (Hsys.preprocess sock dests)
	  )
      | Domain.Mcast(dest,loopback) ->
	  (* Multicast communication mean that this must
	   * be a Deering address.
           *)
	  let sock = if loopback then sock else noloop in
	  let hash = Group.hash_of_id dest in
	  let inet = Hsys.deering_addr hash in 
	  Some (Hsys.preprocess sock [|inet,deering_port|])
      | Domain.Gossip(dest) ->
      	  match mode with
	    (* For gossip messages with Deering, just broadcast
             * it as above.
	     *)
	  | Addr.Deering ->
	      let hash = Group.hash_of_id dest in
	      let inet = Hsys.deering_addr hash in 
	      Some (Hsys.preprocess sock [|inet,deering_port|])

	    (* For Udp, we send the message to the various 
	     * gossip hosts.
	     *)
	  | Addr.Udp ->
	      let dests = Array.map (fun host -> (host,gossip_port)) gossip_hosts in
	      Some (Hsys.preprocess sock dests)
	  | _ -> failwith "bad mode"
    in

    option_map (fun dests ->
      let x = Hsys.sendopt dests in
      let xv = Hsyssupp.sendvopt dests in
      let x,xv =
	if Timestamp.check "send" then (
	  let x buf ofs len = ts_xmit () ; x buf ofs len
	  and xv iov = ts_xmit () ; xv iov
	  in x,xv
	  ) else (x,xv)
      in (x,xv)
    ) dests
  in

  let enable mode group endpt view =
    match mode with
    | Addr.Udp ->
    	Alarm.add_sock alarm sock Hsys.Handler1
    | Addr.Deering ->
      	let hash = Group.hash_of_id group in
      	let deering_sock = Ipmc.join (Hsys.deering_addr hash) deering_port in
    	Alarm.add_sock alarm sock         Hsys.Handler1 ;
      	Alarm.add_sock alarm deering_sock Hsys.Handler1
    | Addr.Sp2 ->
	let sp2_sock = fst (sp2_info ()) in
    	Alarm.add_sock alarm sp2_sock Hsys.Handler1
    | _ -> failwith "enable:bad mode"
  in

  let disable mode group endpt view =
    match mode with
    | Addr.Udp ->
    	Alarm.rmv_sock alarm sock
    | Addr.Deering ->
      	let hash = Group.hash_of_id group in
      	let deering_sock = Ipmc.join (Hsys.deering_addr hash) deering_port in
    	Alarm.rmv_sock alarm sock ;
      	Alarm.rmv_sock alarm deering_sock
    | Addr.Sp2 ->
	let sp2_sock = fst (sp2_info ()) in
    	Alarm.rmv_sock alarm sp2_sock
    | _ -> failwith "disable:bad mode"
  in 

  Domain.create name addr enable disable xmit

(**************************************************************)

let _ = Domain.install Addr.Udp domain

(**************************************************************)
