(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* NETSIM.ML *)
(* Author: Mark Hayden, 4/95 *)
(**************************************************************)
open Util
open Trans
(**************************************************************)
let name = Trace.source_file "NETSIM"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)
(*
let _ =
  Trace.install_root (fun () ->
    eprintf "  NETSIM:alarm:priq:size=%d\n" (Priq.size alarms) ;
(*
    eprintf "    priq:min=%s\n" (Time.to_string (Priq.min alarms)) ;
    List.iter (fun (time,_) ->
      eprintf "      time=%s\n" (Time.to_string time)
    ) (Priq.to_list alarms)
*)
  )
*)
(**************************************************************)

(*
module Priq = Priq.Make ( Time.Ord )
*)

let alarm sched =
  let alarms = 
    let table = Priq.create Time.zero in
    Trace.install_root (fun () ->
      [sprintf "NETSIM:alarms:priq size=%d" (Priq.size table)]
    ) ;
    table
  in
  let time        = ref Time.zero in
  let count       = ref 0 in

  let gettime () = !time in

  let check () =
(*
    eprintf "NETSIM:checking:size=%d:time=%s:min=%s\n" 
      (Priq.size alarms) 
      (Time.to_string !time)
      (Time.to_string (Priq.min alarms)) ;
*)
    Priq.get alarms (fun _ callback ->
      incr count ;
      if !verbose && (!count mod 1000) =| 0 then (
        eprintf "NETSIM:events scheduled=%d, time=%s\n"
      	  !count (Time.to_string !time)
      ) ;
      callback !time
    ) !time
  in
  let min () = Priq.min alarms in

  let alarm callback =
    let disable () = () in
    let schedule time =
      Priq.add alarms time callback
    in Alarm.c_alarm disable schedule
  in

  let little = Time.of_float 0.00001 in
  let block () =
    let newtime =
      let next = Priq.min alarms in
      if Priq.size alarms =| 0 || next < !time then
      	!time
      else
        next
    in
    
    (* Set the time, plus a little.
     *)
    time := Time.add newtime little
  in

  let add_sock _ 	= failwith "add_sock"
  and rmv_sock _ 	= failwith "rmv_sock"
  and add_poll = Alarm.alm_add_poll
  and rmv_poll = Alarm.alm_rmv_poll
  and poll _ = Alarm.alm_poll false
  in Alarm.create
    "NETSIM"
    gettime
    alarm
    check
    min
    add_sock
    rmv_sock
    block
    add_poll
    rmv_poll
    poll

(**************************************************************)

let domain alarm =
  (* This prevents a warning from the Unique module.
   *)
  Unique.install_port (-1) ;

  let msgs = Priq.create Time.zero in
  let ready = Queue.create () in
  let gettime () = Alarm.gettime alarm in

  let poll =
    Mbuf.alloc_dyn name Mbuf.global
    (fun buf ofs len ->
      Priq.get msgs (fun _ msg -> Queue.add msg ready) (gettime ()) ;
      try
      	let msg = Queue.take ready in
	if String.length msg >| len then 
      	  failwith "msg too long" ;
	String.blit msg 0 buf ofs (String.length msg) ;
	String.length msg
      with Queue.Empty -> 0
    )
    (fun iov -> 
      let dint = Iovec.read_int name iov in
      Iovec.break name iov (Route.deliver dint) ;
      Iovec.free name iov
    )
    (fun _ -> failwith "error")
  in

  let addr _ = Addr.NetsimA
  and enable _ _ _ _ = Alarm.add_poll alarm name poll
  and disable _ _ _ _ = Alarm.rmv_poll alarm name
  and xmit _ = function
  | Domain.Pt2pt(a) when a = [||] -> None
  | _ ->
    let x buf ofs len =
      let msg = String.sub buf ofs len in
      let time = Time.add (gettime ()) (Time.of_float 0.01) in
      Priq.add msgs time msg
    in
    let xv iov =
      let iov = Iovecl.flatten name iov in
      Iovec.read name iov x
    in Some(x,xv)
  in

  Domain.create "NETSIM" addr enable disable xmit

(**************************************************************)

let _ =
  Domain.install Addr.Netsim domain ;
  Alarm.install Alarm.Netsim alarm

(**************************************************************)
