(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* IPMC.ML *)
(* Author: Mark Hayden, 5/95 *)
(**************************************************************)
open Util
(**************************************************************)
let name = Trace.source_file "IPMC"
let failwith s = failwith (Util.failmsg name s)
let log = Trace.log name ""
(**************************************************************)
(* Check the installation and print warning messages if not
 * fully configured.  
 *)

let _ =
  if not (Hsys.has_ip_multicast ()) then (
    Trace.config "IPMC" "no header files" ;
    log (fun () -> sprintf "warning, not compiled with IP multicast") ;
    log (fun () -> sprintf "  (couldn't find header files)") ;
  )

(**************************************************************)

let addr () =
  let inet_rand   = Random.int 248 + 7 in
  Hsys.deering_addr inet_rand

(**************************************************************)

let socks = Hashtbl.create 10

let sock where port =
  try Hashtbl.find socks port with
  | Not_found -> (
      let msg = sprintf "illegal private reference to sock in '%s'" where
      in failwith msg
    )

(**************************************************************)

let very_first port () =
  try
    if not (Hsys.has_ip_multicast ()) then (
      eprintf "IPMC:error:not compiled with IP multicast, exiting\n" ;
      exit 1
    ) ;

    (* Open a datagram socket.
     *)
    let sock = Hsys.socket_dgram () in
    log (fun () -> sprintf "sock=%d\n" (Hsys.int_of_socket sock)) ;

    (* Make it reusable.
     *)
    Hsys.setsockopt sock Hsys.Reuse ;

    (* Children don't get access to it and make it non-blocking.
     *)
    (*set_close_on_exec sock ;*)
    (*set_nonblock sock ;*)

    (* Do any other initialization required.  Enable loopback.
     *)
    Hsys.setsockopt sock (Hsys.Multicast true) ;
      
    (* Bind it to the port.
     *)
    Hsys.bind sock (Hsys.inet_any ()) port ;

    (* Save this reference.
     *)
    Hashtbl.add socks port sock

  with e ->
    eprintf "IPMC:unable to initialize IP multicast, exiting\n" ;
    eprintf "  error was:%s\n" (Hsys.error e) ;
    exit 1
  
let very_last port =
  let sock = sock "very_last" port in
    
  Hsys.close sock ;

  (* Remove sock from table.
   *)
  (try
    Hashtbl.remove socks port
  with Not_found -> failwith "very_last:sanity")

let sockets = Resource.create 
  "IPMC:sockets" 
  very_first 
  very_last
  (fun _ -> ())

(**************************************************************)

let first_join (inet,port) () =
  try
    log (fun () -> sprintf "joining:(%s,%d)\n" (Hsys.string_of_inet inet) port) ;
    Hsys.setsockopt (sock "first_join" port) (Hsys.Join inet)
  with e ->
    eprintf "IPMC:joining multicast group:%s, exiting\n" (Hsys.error e) ;
    eprintf "  (this probably means this host does not support IP multicast)\n" ;
    exit 1

let last_leave (inet,port) =
  try
    log (fun () -> sprintf "leaving:(%s,%d)\n" (Hsys.string_of_inet inet) port) ;
    Hsys.setsockopt (sock "last_leave" port) (Hsys.Leave inet)
  with e ->
    eprintf "IPMC:leaving multicast group:%s\n" (Hsys.error e) ;
    exit 1

(**************************************************************)

let groups = Resource.create 
  "IPMC:groups" 
  first_join 
  last_leave
  (fun _ -> ())

let join inet port =
  Resource.add sockets port () ;
  Resource.add groups (inet,port) () ;
  sock "join" port

let leave inet port =
  let sock = sock "leave" port in
  Resource.remove groups (inet,port) ;
  Resource.remove sockets port ;
  sock

(**************************************************************)
