(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* HTK.ML *)
(* Author: Mark Hayden, 3/96 *)
(**************************************************************)
open Tk
open Util
open Trans
open Hsys
(**************************************************************)
let name = Trace.source_file "HTK"
let failwith s = failwith (Util.failmsg name s)
let log = Trace.log name ""
(**************************************************************)

let exec_events () =
  while Sched.step Appl.root_sched 100 do
    log (fun () -> sprintf "looping")
  done

(**************************************************************)

external file_descr_of_socket : 
  socket -> Unix.file_descr = "%identity"

let alarm sched =
  let gettime () = Time.of_float (Hsys.gettimeofday ())

  and alarm callback =
    let disable () = () in
    let schedule time =
      let time 	= Time.to_float time in
      let now 	= Hsys.gettimeofday () in
      let milli	= (time -. now) *. 1000.0 in

      (* We sometimes get timers that are set to go off in
       * in the past!
       *)
      let milli 	= max 0 (truncate milli) in

      add_timer milli (fun () ->
	Printexc.catch (fun () ->
	  log (fun () -> sprintf "timer awake{") ;

	  (* Tk sometimes delivers alarms early, according
	   * got gettimeofday and our calculations above.
	   *)
	  let now = max time (Hsys.gettimeofday ()) in
	  let now = Time.of_float now in

	  (* Make the callback and execute all events to
	   * completion.
	   *)
	  callback now ;
	  exec_events () ;

	  log (fun () -> sprintf "timer asleep}") ;
        ) ()
      ) ; ()
    in Alarm.c_alarm disable schedule
  in

  let sockets = Resource.create "HTK:sockets"

    (* Add a callback for a socket.
     *)
    (fun sock hdlr -> 
      let fd = file_descr_of_socket sock in
      add_fileinput fd (fun () ->
	Printexc.catch (fun () ->
	  log (fun () -> sprintf "sock awake{") ;
	  begin
	    match hdlr with
	    | Hsys.Handler0 f -> f ()
	    | Hsys.Handler1 -> (
		Mbuf.alloc_dyn name Mbuf.global
		  (fun buf ofs len -> Hsys.recv sock buf ofs len)
		  (fun iov ->
		    let dint = Iovec.read_int name iov in
		    Iovec.break name iov (Route.deliver dint)
		  )
		  (fun _ -> ())
		  false ; ()
	      )
	  end ;
          Alarm.alm_poll false ;
	  exec_events () ;
	  log (fun () -> sprintf "sock asleep}") ;
        ) ()
      )
    )

    (* Remove a socket's callback.
     *)
    (fun sock -> 
       let fd = file_descr_of_socket sock in
       remove_fileinput fd)
    (fun _ -> ())
  in

  let add_sock sock hdlr =
    Resource.add sockets sock hdlr

  and remove_sock sock =
    Resource.remove sockets sock

  and min _ 	= failwith "alarm:min"
  and check _ 	= failwith "alarm:check"
  and block _ 	= failwith "alarm:block"
  and poll _	= failwith "alarm:poll"
  and add_poll  = Alarm.alm_add_poll
  and remove_poll = Alarm.alm_rmv_poll

  in Alarm.create
    "HTK"
    gettime
    alarm
    check
    min
    add_sock
    remove_sock
    block
    add_poll
    remove_poll
    poll

let _ = Alarm.install Alarm.Tk alarm

(**************************************************************)

let init () =
  exec_events ()

(**************************************************************)
