(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* ECAMLC.ML *)
(* Author: Mark Hayden, 12/96 *)
(**************************************************************)
open Printf
open Mkutil

let unsafe = ref false
let thread = ref false
let profile = ref false
let assembler = ref false
let destdir = ref None
let includes = ref []
let execdir = Sys.getcwd ()
let debugger = ref false
let ccopts = ref []
let inline = ref ""

let compile srcfile =
  let srcdir  = Filename.dirname srcfile in
  let srcbase = Filename.basename srcfile in
  let srcchop = Filename.chop_extension srcbase in
  let interface = Filename.check_suffix srcfile ".mli" in
  let comp =
    match !optflag, !profile, interface with
    | false, true, false -> "ocamlcp -p fil "
    | false, _, _ -> "ocamlc"
    | true, _, _ -> "ocamlopt"
  in

  let debugger = if !debugger && not !optflag then ["-g"] else [] in
  let inline = if !inline <> "" && !optflag then [!inline] else [] in
  let assembler = if !assembler && !optflag then ["-S"] else [] in
  let unsafe = if !unsafe then ["-unsafe"] else [] in
  let thread = if !thread && not !optflag then ["-thread"] else [] in
  let output = if platform = Unix then ["1> .err 2> .err"] else ["> .err"] in
  let includes = List.map (fun dir -> sprintf "-I %s" (Filename.concat execdir dir)) !includes in
  let com = [
    [comp ; "-c"] @
    inline @
    assembler @
    debugger @
    unsafe @
    inline @
    thread @
    !ccopts @
    includes @
    [srcbase] @
    output
  ] in

  let com = String.concat " " (List.flatten com) in 

  (* Change to the source directory, compile, and change back.
   *)
  Sys.chdir srcdir ;
  if !verbose then (
    eprintf "ecamlc: %s\n" com ;
    flush stderr
  ) ;
  let ret = Sys.command com in
  Sys.chdir execdir ;

(*  if ret <> 0 then *)  (
    let errout = Filename.concat srcdir ".err" in
    if !verbose then (
      eprintf "ecamlc: compiler returned error code %d\n" ret ;
      eprintf "ecamlc: reading error file %s\n" errout ;
      flush stderr
    ) ;
    if Sys.file_exists errout then (
      let ch = open_in errout in
      (try while true do
        let line = input_line ch in 
	let line = global_replace line srcbase srcfile in
	let line =
	  if am_hayden () then (
	    let eroot = "/usr/u/hayden/ensemble/" in
	    let line = global_replace line "../" eroot in
	    let line = global_replace line "/amd/gulag/a/hayden/" eroot in
	    let line = global_replace line "/amd/sunup/a/hayden/" eroot in
	    line
	  ) else (
	    line
	  )
	in
        print_string line ;
        print_newline ()
      done with End_of_file -> ())
    ) ;
(*
    if !verbose then
      eprintf "ecamlc: exiting\n"
*)
  ) ;

  begin
    let efile_exists f =
      if Sys.file_exists f then (
      	let ch = open_in f in	
      	let len = in_channel_length ch in
	close_in ch ;
	if len = 0 then (
	  Sys.remove f ;
	  false
	) else true
      ) else false
    in

    match !destdir with
    | None -> ()
    | Some destdir -> (
	List.iter (fun ext ->
      	  let exd = plat_ext ext in
	  let src = Filename.concat srcdir (srcchop ^ ext) in
	  let dst = Filename.concat destdir (srcchop ^ exd) in
(*
	  let src = hayden src in
	  let dst = hayden dst in
*)

	  if efile_exists src && src <> dst then (
	    if !verbose then 
	      eprintf "ecamlc: moving %s %s\n" src dst ;
	    if Sys.file_exists dst then
	      Sys.remove dst ;

            Sys.catch_break true ;
      	    begin try 
	      if !verbose then
		eprintf "ecamlc: actual rename being done\n" ;
      	      Sys.rename src dst ;
	      
	      (* For cmx files, also make a .cmx copy (for inlining).
	       *)
	      (* This is a total hack: prevent there from being
	       * a socket.cmx so that inlining cannot be done against
	       * it.
	       *)
	      if ext = ".cmx" && srcchop <> "socket" then 
		copy dst (Filename.concat destdir (srcchop ^ ext)) ;

      	      Sys.catch_break false
      	    with
	    | Sys.Break ->
	        if Sys.file_exists dst then
		  Sys.remove dst ;
		exit 1
	    | e ->
	        Sys.catch_break false ;
      	        raise e 
	    end
	  )
	) (oc_extc ())
      )
  end ;

  if ret <> 0 then 
    exit ret

  

let main () =
  try
    Arg.parse
      ["-I", Arg.String(fun dir -> includes := dir :: !includes),"";
       "-o", Arg.String(fun dir -> destdir := Some dir),"";
       "-S", 		Arg.Set assembler,"" ;
       "-g",            Arg.Set debugger,"" ;
       "-verbose",      Arg.Set verbose,"" ;
       "-unsafe",	Arg.Set unsafe,"" ;
       "-thread",       Arg.Set thread,"" ;
       "-profile",      Arg.Set profile,"" ;
       "-plat",         Arg.String(fun s -> plat := "-"^s),"" ;
       "-inline",       Arg.Int(fun i -> inline := sprintf "-inline %d" i),"" ;
       "-ccopt",        Arg.String(fun s -> ccopts := !ccopts @ [" -ccopt "^s]),"" ;
       "-opt",          Arg.Set optflag,""]
      (fun file -> compile file)
      ""
  with x -> raise x

let _ = 
  Printexc.catch main () ;
  exit 0
