/**************************************************************/
/*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 */
/**************************************************************/
/***********************************************************************/
/*                                                                     */
/*  Written by Robbert van Renesse, Cornell University. 	       */
/*                                                                     */
/***********************************************************************/

#include <stdio.h>
#include <assert.h>
#include "skt.h"

#ifdef HAS_SOCKETS

enum { E_WEXITED, E_WSIGNALED, E_WSTOPPED };

#ifdef _WINSOCKAPI_

#include <process.h>

/* This is a process descriptor that is passed to the threads that
 * wait for them.
 */
struct proc_descr {
	PROCESS_INFORMATION pi;
	int socket;
};

/* This is an event that is written to the socket specified in the
 * process descriptor.
 */
struct event {
	struct proc_descr *pd;
};


#else /* !WINSOCK, i.e., UNIX */

#include <signal.h>

/* This is a process descriptor.
 */
struct proc_descr {
	int pid;
};

/* This is an event that is written to the socket.
 */
struct event {
	struct proc_descr *pd;
	int pid, status;
};

#endif /* WINSOCK */

/* Read a process event of the given socket.
 */
static void await_event(int skt, struct event *ev){
	int n, total = 0;

	while ((n = recv(skt, ((char *) ev) + total,
			sizeof(*ev) - total, 0)) > 0) {
		total += n;
		if (total == sizeof(*ev))
			return;
	}
	uerror("await_event", Nothing);
	exit(1);	
}

#ifdef _WINSOCKAPI_

/* This is the thread that actually spawns the process, waits for its
 * termination, and sends the message to the given socket.
 */
static DWORD skt_spawn_thread(LPDWORD p){
	struct proc_descr *pd = (struct proc_descr *) p;
	struct event ev;
	struct sockaddr skt_addr;
	int len, n;

	len = sizeof(skt_addr);
	(void) getsockname(pd->socket, &skt_addr, &len);
	ev.pd = pd;
	if (WaitForSingleObject(pd->pi.hProcess, INFINITE) !=
						WAIT_OBJECT_0) {
		printf("WaitForSingleObject failed!\n");
		exit(1);
	}
	n = sendto(pd->socket, (char *) &ev, sizeof(ev), 0, &skt_addr, len);
	if (n <= 0)
		printf("sendto --> %d %d\n", n, h_errno);
	ExitThread(0);
	return 0;
}

/* Make a command line out of the given arguments.
 */
static char *cmdline(value args){
	char *res, *p;
	int size, total = 0, i;

	size = Wosize_val(args);
	for (i = 0; i < size; i++)
		total += string_length(Field(args, i)) + 1;
	res = p = (char *) stat_alloc(total);
	for (i = 0; i < size; i++) {
		strcpy(p, String_val(Field(args, i)));
		p += string_length(Field(args, i));
		*p++ = ' ';
	}
	*--p = 0;
	return res;
}

/* Create a new process, and return a process handle.
 * When some event happens to the process that should be notified (e.g.,
 * death), report it to the given socket.
 *
 * We create a thread that waits for termination of the process.
 */
value skt_spawn_process(
	value cmd,
	value args,
	value skt
) {
	extern char *searchpath(char *);
	STARTUPINFO si;
	char *exefile, *argv;
	struct proc_descr *pd;
	DWORD tid;

	exefile = searchpath(String_val(cmd));
	if (exefile == 0)
		exefile = String_val(cmd);
	argv = cmdline(args);
	GetStartupInfo(&si);
	pd = (struct proc_descr *) stat_alloc(sizeof(*pd));
	if (!CreateProcess(exefile, argv, 0, 0,
                	TRUE, 0, 0, 0, &si, &pd->pi)) {
		_dosmaperr(GetLastError());
		uerror("skt_spawn_process", exefile);
	}
	stat_free(argv);
	pd->socket = Int_val(skt);
	CreateThread(0, 0, (LPTHREAD_START_ROUTINE) skt_spawn_thread,
			(LPVOID) pd, 0, &tid);
	return (value) pd;	
}

/* Terminate the given process.
 */
value skt_terminate_process(value ph){
	struct proc_descr *pd = (struct proc_descr *) ph;

	TerminateProcess(pd->pi.hProcess, 0x10000009);
	return Val_unit;
}

/* Read a process termination event of the given socket and
 * return the process handle and the termination status.  After
 * this, the process handle is no longer valid.
 */
value skt_wait_process(value skt){
	struct event ev;
	int status;
	value result;
	Push_roots(r, 1);

	await_event(Int_val(skt), &ev);
	if (!GetExitCodeProcess(ev.pd->pi.hProcess, &status) ||
					status == STILL_ACTIVE) {
		printf("GetExitCode failed!\n");
		exit(1);
	}
	if ((status & 0xFFFF0000) == 0x10000000) {
		r[0] = alloc(1, E_WSIGNALED);
		Field(r[0], 0) = Val_int(status & 0xFF);
	}
	else {
		r[0] = alloc(1, E_WEXITED);
		Field(r[0], 0) = Val_int(status);
	}
	result = alloc_tuple(2);
	Field(result, 0) = (value) ev.pd;
	Field(result, 1) = r[0];
	stat_free((char *) ev.pd);
	Pop_roots();
	return result;
}

#else /* !WINSOCK, i.e., UNIX */

/* The following bit is stolen from the Ocaml library.
 */

#include <sys/wait.h>

#if !(defined(WIFEXITED) && defined(WEXITSTATUS) && defined(WIFSTOPPED) && \
      defined(WSTOPSIG) && defined(WTERMSIG))
#define WIFEXITED(status) ((status) & 0xFF == 0)
#define WEXITSTATUS(status) (((status) >> 8) & 0xFF)
#define WIFSTOPPED(status) ((status) & 0xFF == 0xFF)
#define WSTOPSIG(status) (((status) >> 8) & 0xFF)
#define WTERMSIG(status) ((status) & 0x3F)
#endif

static value alloc_process_status(struct proc_descr *pd, int status){
	value st, res;
	Push_roots(r, 1);

	if (WIFEXITED(status)) {
		st = alloc(1, E_WEXITED);
		Field(st, 0) = Val_int(WEXITSTATUS(status));
		stat_free((char *) pd);
	}
	else if (WIFSTOPPED(status)) {
		st = alloc(1, E_WSTOPPED);
		Field(st, 0) = Val_int(WSTOPSIG(status));
	}
	else {
		st = alloc(1, E_WSIGNALED);
		Field(st, 0) = Val_int(WTERMSIG(status));
		stat_free((char *) pd);
	}
	r[0] = st;
	res = alloc_tuple(2);
	Field(res, 0) = (value) pd;
	Field(res, 1) = r[0];
	Pop_roots();
	return res;
}

/* Send an event to the given socket.
 */
static void notify(int skt, struct event *ev){
	struct sockaddr skt_addr;
	int len;

	len = sizeof(skt_addr);
	(void) getsockname(skt, &skt_addr, &len);
	/*printf("sending %d bytes\n", sizeof(*ev));*/
	if (sendto(skt, (void*)ev, sizeof(*ev), 0, &skt_addr, len) != sizeof(*ev))
		perror("skt_spawn_process: notify: sendto");
}

extern char ** cstringvect();	/* From O'Caml Unix library */

/* Create a new process, and return a process handle.
 * When some event happens to the process that should be notified (e.g.,
 * death), report it to the given socket.
 *
 * We fork twice (nested).  The first process forked waits for the second,
 * and does the notification.
 */
value skt_spawn_process(
	value cmd,
	value args,
	value skt
) {
	extern char *searchpath(char *);
	char *exefile, **argv;
	struct proc_descr *pd;
	int i, pid, fd = Int_val(skt);
	struct event ev;

	exefile = String_val(cmd);
	pd = (struct proc_descr *) stat_alloc(sizeof(*pd));
	switch (fork()) {
	case -1:
		uerror("skt_spawn_process", cmd);
		break ;
	case 0:
		pid = fork();
		switch (pid) {
		case -1:
			perror("skt_spawn_process: fork");
			notify(fd, &ev);
			exit(1);
		case 0:
			for (i = 3; i < 64; i++)
				close(fd);
			argv = cstringvect(args);
			execv(exefile, argv);
			perror(exefile);
			exit(1);
		default:
			ev.pd = pd;
			ev.pid = pid;
			notify(fd, &ev);
			if (wait(&ev.status) != pid)
				perror("skt_spawn_process: wait");
			notify(fd, &ev);
			exit(0);
		}
		break ;
	default:
		await_event(fd, &ev);
		pd->pid = ev.pid;
		return (value) pd;
		break ;
	}
	assert(0) ;
}

/* Terminate the given process.
 */
value skt_terminate_process(value ph){
	struct proc_descr *pd = (struct proc_descr *) ph;

	kill(pd->pid, SIGKILL);
	return Val_unit;
}

/* Read a process termination event of the given socket and
 * return the process handle and the termination status.  After
 * this, the process handle is no longer valid.
 */
value skt_wait_process(value skt){
	struct event ev;
	int dummy;

	await_event(Int_val(skt), &ev);
	/* The typecasts to void* are for
	 * AIX.
	 */
	while (wait3((void*)&dummy, WNOHANG, (void*)/*(struct rusage *)*/ NULL) > 0)
	  /* clean up zombies */;
	return alloc_process_status(ev.pd, ev.status);
}

#endif /* WINSOCK */

#else /* !HAS_SOCKETS */

value skt_spawn_process(value cmd, value cmdline, value skt){
	failwith("skt_spawn_process: not available");
}

value skt_terminate_process(value ph){
	failwith("skt_terminate_process: not available");
}

value skt_wait_process(value skt){
	failwith("skt_terminate_process: not available");
}

#endif /* HAS_SOCKETS */

