(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* SOCKET.MLI *)
(* Authors: Robbert vanRenesse and Mark Hayden, 4/95 *)
(**************************************************************)

type buf = string
type ofs = int
type len = int

type info

type iovec = buf * ofs * len

type handler =
  | Handler0 of (unit -> unit)
  | Handler1

type manager

(**************************************************************)

val stdin : unit -> Unix.file_descr	(* really a socket *)

(* For reading from stdin.
 *)
val read : Unix.file_descr -> buf -> ofs -> len -> len

(**************************************************************)

(*** Polling. *)

val select :
  Unix.file_descr list -> Unix.file_descr list -> Unix.file_descr list -> float ->
        Unix.file_descr list * Unix.file_descr list * Unix.file_descr list 
        (* Wait until some input/output operations become possible on
           some channels. The three list arguments are, respectively, a set
           of descriptors to check for reading (first argument), for writing
           (second argument), or for exceptional conditions (third argument).
           The fourth argument is the maximal timeout, in seconds; a
           negative fourth argument means no timeout (unbounded wait).
           The result is composed of three sets of descriptors: those ready
           for reading (first component), ready for writing (second component),
           and over which an exceptional condition is pending (third
           component). *)

(**************************************************************)

(* Allocate a static string (with stat_alloc).
 *)
val static_string : len -> string

(* Release such a string (with stat_free).
 *)
val static_string_free : string -> unit

(**************************************************************)

type manager
val manager : (Unix.file_descr * handler) list -> len -> (int -> len -> unit) -> manager
val poll : manager -> buf -> ofs -> bool

(**************************************************************)

(* Optimized versions of above functions.  No exceptions are
 * raised.  Scatter-gather used if possible.  No interrupts
 * are serviced during call.  No bounds checks are made.
 * Errors are returned as -1.  Call errno or perror to get
 * exact error code.  Probably not threadsafe.  *)

val preprocess : Unix.file_descr -> Unix.msg_flag list -> Unix.sockaddr array -> info

(* BUG: these probably should not be marked noalloc (because
 * if they are not defined, they will raise an exception).
 *)
val sendopt : info -> buf -> ofs -> len -> unit
val sendvopt : info -> iovec array -> unit

(**************************************************************)

(*** Multicast support. *)

val has_ip_multicast : unit -> bool 
      	(* Return whether this host supports IP multicast. *)
val setsock_multicast : Unix.file_descr -> bool -> unit 
      	(* Setup a socket for multicast operation. *)
val setsock_join : Unix.file_descr -> Unix.inet_addr -> unit 
      	(* Join an IP multicast group. *)
val setsock_leave : Unix.file_descr -> Unix.inet_addr -> unit 
        (* Leave an IP multicast group. *)
val setsock_sendbuf : Unix.file_descr -> int -> unit
  
val setsock_recvbuf : Unix.file_descr -> int -> unit

(**************************************************************)

(* HACK!  It's useful to be able to print these out as ints.
 *)
val int_of_file_descr : Unix.file_descr -> int

(**************************************************************)
(* PUSH_INT/POP_INT: read and write 4-byte integers to
 * strings in network byte order 
 *)

val push_int : string -> int -> int -> unit 
val pop_int  : string -> int -> int

(**************************************************************)

(* Same as Unix.gettimeofday.  Except that
 * the caller passes in a 1-element double array.
 * This prevents allocation for native code.
 *)
val gettimeofday : float array -> unit

(**************************************************************)

(* MARSHAL: [marshal o buf ofs len] Marshals [o] into the
 * given portion of buf deliminated ofs and len.  Returns
 * the number of bytes written to the string.  If the object
 * is too large, then a negative value is returned.
 * Regardless of value returned, the bytes of buf
 * deliminated by ofs and len may have been modified.
 *
 * Other notes: 
 *
 * + ofs & len should always be multiples of 4.
 *
 * + return value is either negative (failure) or multiple of 4
 *)
val marshal : Obj.t -> buf -> ofs -> len -> len

(**************************************************************)

type process_handle

val process_socket : unit -> Unix.file_descr
val spawn_process : string -> string array -> Unix.file_descr -> process_handle
val wait_process : Unix.file_descr -> (process_handle * Unix.process_status)
val terminate_process : process_handle -> unit

(**************************************************************)

val recv : Unix.file_descr -> buf -> ofs -> len -> len
val send : Unix.file_descr -> buf -> ofs -> len -> len

(**************************************************************)

val fork : unit -> int

(**************************************************************)
