(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* SOCKET.MLI *)
(* Authors: Robbert vanRenesse and Mark Hayden, 4/95 *)
(**************************************************************)

type buf = string
type ofs = int
type len = int

type info

type iovec = buf * ofs * len

type handler =
  | Handler0 of (unit -> unit)
  | Handler1

type socket				(* Not exported *)

type manager = {
  socks 	: socket array ;
  handlers 	: handler array ;
  max_len       : len ;
  handler       : int -> len -> unit ;
  raw : (Unix.file_descr * handler) list
}

(**************************************************************)

let trace = 
  try 
    Sys.getenv "SOCKET_TRACE" ; 
    true 
  with Not_found -> 
    false

let debug_msg s = 
  if trace then (
    Printf.fprintf Pervasives.stderr "SOCKET:trace:%s\n" s ;
    flush Pervasives.stderr
  )

(**************************************************************)

let is_unix =
  match Sys.os_type with
  | "Unix" -> true
  | "Win32" -> false
  | s -> 
      Printf.fprintf Pervasives.stderr "SOCKET:get_config:failed:os=%s\n" s ;
      flush Pervasives.stderr ;
      exit 1
  
(**************************************************************)

external fork : unit -> int = "skt_fork"

external gettimeofday : float array -> unit = "skt_gettimeofday" "skt_gettimeofday_native"

external socket_of_fd : Unix.file_descr -> socket = "skt_socket_of_fd"

(**************************************************************)

external start_input : unit -> Unix.file_descr = "skt_start_input"

(* See documentation in stdin.c.
 *)
let stdin =
  let stdin = ref None in
  if is_unix then (
    fun () -> Unix.stdin 
  ) else (
    fun () ->
      match !stdin with
      | None ->
	  let s = start_input () in
	  stdin := Some s ;
	  s
      | Some s -> s
  )

(* READ: Depending on the system, call either read or recv.
 * This is only used for stdin (where on Nt it is a socket).
 *)
let read =
  if is_unix then 
    Unix.read
  else
    fun s b o l -> fst (Unix.recvfrom s b o l [])

(**************************************************************)

external select :
  socket list -> socket list -> socket list -> float ->
        socket list * socket list * socket list 
  = "skt_select"

(**************************************************************)

external static_string : len -> string = "skt_static_string"
external static_string_free : string -> unit = "skt_static_string_free"

(**************************************************************)

let manager info max_len handler =
  let socks,handlers = List.split info in
  let socks = Array.of_list socks in
  let socks = Array.map socket_of_fd socks in
  let handlers = Array.of_list handlers in
  
  { socks = socks ;
    handlers = handlers ;
    max_len = max_len ;
    handler = handler  ; 
    raw = info } 

external poll : manager -> buf -> ofs -> bool = "skt_poll"

(**************************************************************)
(* Optimized socket operations.
 * - unsafe
 * - no exceptions
 * - "noalloc" is enabled
 * - socket, flags, and address are preprocessed
 * - no interrupts accepted
 * - no retransmission attempts on interrupts
 *)

external preprocess : socket -> Unix.msg_flag list -> Unix.sockaddr array -> info
  = "skt_preprocess"
external sendopt : info -> buf -> ofs -> len -> unit
  = "skt_sendopt" "noalloc"
external sendvopt : info -> iovec array -> unit
  = "skt_sendvopt" "noalloc"

(**************************************************************)

external has_ip_multicast : unit -> bool 
  = "skt_has_ip_multicast" 
external setsock_multicast : socket -> bool -> unit 
  = "skt_setsock_multicast" 
external setsock_join : socket -> Unix.inet_addr -> unit 
  = "skt_setsock_join" 
external setsock_leave : socket -> Unix.inet_addr -> unit 
  = "skt_setsock_leave" 
external setsock_sendbuf : socket -> int -> unit
  = "skt_setsock_sendbuf"
external setsock_recvbuf : socket -> int -> unit
  = "skt_setsock_recvbuf"

(**************************************************************)

(* HACK!  It's useful to be able to print these out as ints.
 *)
external int_of_file_descr : socket -> int 
  = "%identity"

(**************************************************************)
(* PUSH_INT/POP_INT: read and write 4-byte integers to
 * strings in network byte order 
 *)

external push_int : string -> ofs -> int -> unit 
  = "skt_push_int" "noalloc"
external pop_int  : string -> ofs -> int
  = "skt_pop_int" "noalloc"

(**************************************************************)
(* MARSHAL: [marshal o buf ofs len] Marshals [o] into the
 * given portion of buf deliminated ofs and len.  Returns
 * the number of bytes written to the string.  If the object
 * is too large, then a negative value is returned.
 * Regardless of value returned, the bytes of buf
 * deliminated by ofs and len may have been modified.  
 * Compatible with Obj.marshal and Obj.unmarshal.
 *)
external marshal: Obj.t -> buf -> ofs -> len -> len = "skt_marshal" "noalloc"

(**************************************************************)

type process_handle

(* Create a socket that can be used in spawn_process().  It is
 * used to receive termination events on.
 *)
let process_socket () =
  let sock = Unix.socket Unix.PF_INET Unix.SOCK_DGRAM 0 in
  let host = Unix.gethostname() in
  let h = Unix.gethostbyname host in
  let addr = h.Unix.h_addr_list.(0) in
  Unix.bind sock (Unix.ADDR_INET(addr, 0));
  sock

(* Usage: spawn_process "name-of-executable" arguments socket.
 * E.g.: spawn_process "echo.exe" "echo hello world" skt.
 * Returns a handle on the process, and eventually sends a termination
 * event to the given socket.
 *)
external spawn_process : string -> string array -> socket -> process_handle
  = "skt_spawn_process"

(* Returns the process handle and the termination status of the next
 * process (that was started using spawn_process on the given socket)
 * to terminate.  Unless the process_status is WSTOPPED, the process
 * handle is no longer valid after this.
 *)
external wait_process : socket -> (process_handle * Unix.process_status)
  = "skt_wait_process"

(* Terminates the given process immediately.
 *)
external terminate_process : process_handle -> unit
  = "skt_terminate_process"

(**************************************************************)

external recv : socket -> buf -> ofs -> len -> len = "skt_recv"
external send : socket -> buf -> ofs -> len -> len = "skt_send"

(**************************************************************)
(* Here we wrap all of the functions with calls to
 * socket_of_fd.  This deals with the OSF socket handles.
 *)

let preprocess s m a =
  let s = socket_of_fd s in
  preprocess s m a

let setsock_multicast s b =
  let s = socket_of_fd s in
  setsock_multicast s b

let setsock_join s a =
  let s = socket_of_fd s in
  setsock_join s a

let setsock_leave s a =
  let s = socket_of_fd s in
  setsock_leave s a

let setsock_sendbuf s l =
  let s = socket_of_fd s in
  setsock_sendbuf s l

let setsock_recvbuf s l =
  let s = socket_of_fd s in
  setsock_recvbuf s l

let int_of_file_descr s =
  let s = socket_of_fd s in
  int_of_file_descr s

let spawn_process path args sock =
  let sock = socket_of_fd sock in
  spawn_process path args sock

let wait_process sock =
  let sock = socket_of_fd sock in
  wait_process sock

let send s b o l =
  let s = socket_of_fd s in
  send s b o l

let recv s b o l =
  let s = socket_of_fd s in
  recv s b o l

let ss socks fds =
  let socks = List.map (fun fd -> (socket_of_fd fd),fd) socks in
  List.map (fun sock -> List.assoc sock socks) fds

let select a b c d =
  let a' = List.map socket_of_fd a in
  let b' = List.map socket_of_fd b in
  let c' = List.map socket_of_fd c in
  let (e,f,g) = select a' b' c' d in
  let e = ss a e in
  let f = ss b f in
  let g = ss c g in
  (e,f,g)

(**************************************************************)

let unix_wrap debug f =
  try f () with Unix.Unix_error(err,s1,s2) as exc ->
    match err with 
    | Unix.ECONNREFUSED 
    | Unix.ECONNRESET 
    | Unix.EPIPE 
    | Unix.ENETUNREACH
    | Unix.EAGAIN -> 0
    | _ ->
	Printf.eprintf "SOCKET:%s:%s\n" debug (Unix.error_message err) ;
	  flush stderr ;
	  raise exc

let recv s b o l = unix_wrap "recv" (fun () -> recv s b o l)

(*
let poll m buf ofs =
  let socks = List.map fst m.raw in
  let avail,_,_ = select socks [] [] 0.0 in
  if avail = [] then false else (
    let sock = List.hd avail in
    begin
      match List.assq sock m.raw with
      | Handler0(f) -> f ()
      | Handler1 ->
 	  let len = recv sock buf ofs m.max_len in
	  let dint = pop_int buf ofs in
	  m.handler dint len
    end ;
    true
  )
*)
(**************************************************************)

let _ = debug_msg "socket_done"

(**************************************************************)
