/**************************************************************/
/*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 */
/**************************************************************/
/**************************************************************/
/* MARSHAL.C */
/* Author: Mark Hayden, 5/96 */
/**************************************************************/
/* Based on byterun/extern.c in Objective Caml */
/**************************************************************/

/***********************************************************************/
/*                                                                     */
/*                           Objective Caml                            */
/*                                                                     */
/*            Xavier Leroy, projet Cristal, INRIA Rocquencourt         */
/*                                                                     */
/*  Copyright 1996 Institut National de Recherche en Informatique et   */
/*  Automatique.  Distributed only by permission.                      */
/*                                                                     */
/***********************************************************************/

/* $Id: marshal.c,v 1.3 1997/04/29 14:41:17 hayden Exp $ */

/* Structured output */

#include <string.h>
#include <stdio.h>
#include <assert.h>
#include "skt.h"
#include "alloc.h"
#include "include/gc.h"		/* for Black */
#include "include/intext.h"	/* for magic numbers */
#include "include/freelist.h"   /* required for major_gc.h */
#include "include/major_gc.h"   /* for Is_in_heap */
#include "include/minor_gc.h"   /* for Is_young */
#include "memory.h"
#include "misc.h"
#include "mlvalues.h"

/* To keep track of sharing in externed objects */

typedef unsigned long byteoffset_t;

struct extern_obj {
  value obj ;
  byteoffset_t ofs ;
  struct extern_obj *next ;		/* linked list of entries */
} ;

static struct extern_obj * extern_table = NULL ;
static unsigned long extern_table_size ;
static struct extern_obj * extern_next = NULL ;

#ifdef ARCH_SIXTYFOUR
#define Hash(v) (((unsigned long) ((v) >> 3)) % extern_table_size)
#else
#define Hash(v) (((unsigned long) ((v) >> 2)) % extern_table_size)
#endif

/* Allocate hash table if not already around.
 */
static void alloc_extern_table(void) {
  asize_t i;

  if (!extern_table) {
    extern_table_size = INITIAL_EXTERN_TABLE_SIZE ;
    extern_table = (struct extern_obj *)
      stat_alloc(extern_table_size * sizeof(struct extern_obj));
    for (i = 0; i < extern_table_size; i++)
      extern_table[i].obj = 0 ;
    extern_next = NULL ;
  }
}

/* Double size of hash table.
 */
static void resize_extern_table(void) {
  struct extern_obj *old, *old_table ;
  asize_t i, h;
  value obj;

  old_table = extern_table ;
  old = extern_next;

  /* Double the table size.
   */
  extern_table_size = 2 * extern_table_size;

  /* Allocate the new table and zero the obj pointers.
   */
  extern_table = (struct extern_obj *)
    stat_alloc(extern_table_size * sizeof(struct extern_obj));
  extern_next = NULL ;
  for (i = 0; i < extern_table_size; i++)
    extern_table[i].obj = 0 ;

  /* Follow old list of pointers to copy over the data.
   */
  for (;old; old=old->next) {
    obj = old->obj ;
    assert(obj) ;
    h = Hash(obj) ;
    while (extern_table[h].obj != 0) {
      h++ ;
      if (h >= extern_table_size) h = 0 ;
    }
    extern_table[h].obj = obj ;
    extern_table[h].ofs = old->ofs ;
    extern_table[h].next = extern_next ;
    extern_next = &extern_table[h] ;
  }    

  /* Finally, release the old table.
   */
  stat_free((char *) old_table);
}

/* If larger than initial size then release the table.
 * Otherwise, use the links to zero the table entries in
 * preparation for the next call.  
 */
static void free_extern_table(void) {
  if (extern_table_size > INITIAL_EXTERN_TABLE_SIZE) {
    stat_free((char *) extern_table) ;
    extern_table = NULL ;
    extern_next = NULL ;
  } else {
    for (;extern_next; extern_next = extern_next->next)
      extern_next->obj = 0 ;
    assert(!extern_next) ;
  }
}

/* To buffer the output.
 */
static char *extern_block, *extern_ptr, *extern_limit;
static int error_occurred ;

static void out_of_space(void) {
  error_occurred = 1 ;
}

#define Write(c)                   \
  if (extern_ptr >= extern_limit)  \
    out_of_space();                \
  else                             \
  *extern_ptr++ = (c)

/* Write integers and blocks in the output buffer */

static void writeblock(data, len)
     char * data;
     long len;
{
  if (extern_ptr + len > extern_limit) out_of_space();
  else {
    bcopy(data, extern_ptr, len);
    extern_ptr += len;
  }
}

static void writecode8(code, val)
     int code;
     long val;
{
  if (extern_ptr + 2 > extern_limit) out_of_space();
  else {
    extern_ptr[0] = code;
    extern_ptr[1] = val;
    extern_ptr += 2;
  }
}

static void writecode16(code, val)
     int code;
     long val;
{
  if (extern_ptr + 3 > extern_limit) out_of_space();
  else {
    extern_ptr[0] = code;
    extern_ptr[1] = val >> 8;
    extern_ptr[2] = val;
    extern_ptr += 3;
  }
}

static void write32(val)
     long val;
{
  if (extern_ptr + 4 > extern_limit) out_of_space();
  else {
    extern_ptr[0] = val >> 24;
    extern_ptr[1] = val >> 16;
    extern_ptr[2] = val >> 8;
    extern_ptr[3] = val;
    extern_ptr += 4;
  }
}

static void writecode32(code, val)
     int code;
     long val;
{
  if (extern_ptr + 5 > extern_limit) out_of_space();
  else {
    extern_ptr[0] = code;
    extern_ptr[1] = val >> 24;
    extern_ptr[2] = val >> 16;
    extern_ptr[3] = val >> 8;
    extern_ptr[4] = val;
    extern_ptr += 5;
  }
}

#ifdef ARCH_SIXTYFOUR
static void writecode64(code, val)
     int code;
     long val;
{
  int i;
  if (extern_ptr + 9 > extern_limit) out_of_space();
  else {
    *extern_ptr ++ = code;
    for (i = 64 - 8; i >= 0; i -= 8) *extern_ptr++ = val >> i;
  }
}
#endif

/* Marshal the given value in the output buffer */

static byteoffset_t obj_counter;    /* Number of objects emitted so far */
static unsigned long size_32;  /* Size in words of 32-bit block for struct. */
static unsigned long size_64;  /* Size in words of 64-bit block for struct. */

static void marshal_rec(v)
     value v;
{
 tailcall:
  if (Is_long(v)) {
    long n = Long_val(v);
    if (n >= 0 && n < 0x40) {
      Write(PREFIX_SMALL_INT + n);
    } else if (n >= -(1 << 7) && n < (1 << 7)) {
      writecode8(CODE_INT8, n);
    } else if (n >= -(1 << 15) && n < (1 << 15)) {
      writecode16(CODE_INT16, n);
#ifdef ARCH_SIXTYFOUR
    } else if (n < -(1L << 31) || n >= (1L << 31)) {
      writecode64(CODE_INT64, n);
#endif
    } else
      writecode32(CODE_INT32, n);
#ifdef DO_NOT_INCLUDE

* This code does not work with native code unless compiled
* with NATIVE_CODE flag

  } else if (!Is_atom(v) && !Is_young(v) && !Is_in_heap(v)) {
    invalid_argument("marshal: abstract value(2)");
#endif
  } else {
    header_t hd = Hd_val(v);
    tag_t tag = Tag_hd(hd);
    mlsize_t sz = Wosize_hd(hd);
    asize_t h;
    /* Atoms are treated specially for two reasons: they are not allocated
       in the externed block, and they are automatically shared. */
    if (sz == 0) {
      if (tag < 16) {
        Write(PREFIX_SMALL_BLOCK + tag);
      } else {
        writecode32(CODE_BLOCK32, hd);
      }
    } else {
      /* Check if already seen */
      h = Hash(v);
      while (extern_table[h].obj != 0) {
        if (extern_table[h].obj == v) {
          byteoffset_t d = obj_counter - extern_table[h].ofs;
          if (d < 0x100) {
            writecode8(CODE_SHARED8, d);
          } else if (d < 0x10000) {
            writecode16(CODE_SHARED16, d);
          } else {
            writecode32(CODE_SHARED32, d);
          }
          return;
        }
        h++;
        if (h >= extern_table_size) h = 0;
      }
      /* Not seen yet. Record the object and output its contents. */
      extern_table[h].obj = v ;
      extern_table[h].ofs = obj_counter ;
      extern_table[h].next = extern_next ;
      extern_next = &extern_table[h] ;

      obj_counter++;
      if (2 * obj_counter >= extern_table_size) 
	resize_extern_table();

      switch(tag) {
      case String_tag: {
        mlsize_t len = string_length(v);
        if (len < 0x20) {
          Write(PREFIX_SMALL_STRING + len);
        } else if (len < 0x100) {
          writecode8(CODE_STRING8, len);
        } else {
          writecode32(CODE_STRING32, len);
        }
        writeblock(String_val(v), len);
        size_32 += 1 + (len + 4) / 4;
        size_64 += 1 + (len + 8) / 8;
        break;
      }
      case Double_tag: {
        if (sizeof(double) != 8) {
          free_extern_table();
          invalid_argument("marshal: non-standard floats");
        }
        Write(CODE_DOUBLE_NATIVE);
        writeblock((char *) v, 8);
        size_32 += 1 + 2;
        size_64 += 1 + 1;
        break;
      }
      case Double_array_tag: {
        mlsize_t nfloats;
        if (sizeof(double) != 8) {
          free_extern_table();
          invalid_argument("marshal: non-standard floats");
        }
        nfloats = Wosize_val(v) / Double_wosize;
        if (nfloats < 0x100) {
          writecode8(CODE_DOUBLE_ARRAY8_NATIVE, nfloats);
        } else {
          writecode32(CODE_DOUBLE_ARRAY32_NATIVE, nfloats);
        }
        writeblock((char *) v, Bosize_val(v));
        size_32 += 1 + nfloats * 2;
        size_64 += 1 + nfloats;
        break;
      }
      case Abstract_tag:
      case Final_tag:
        free_extern_table();
        invalid_argument("marshal: abstract value(1)");
        break;
      case Closure_tag:
      case Infix_tag:
        free_extern_table();
        invalid_argument("marshal: functional value");
        break;
      default: {
        mlsize_t i;
        if (tag < 16 && sz < 8) {
          Write(PREFIX_SMALL_BLOCK + tag + (sz << 4));
        } else {
          writecode32(CODE_BLOCK32, hd & ~Black);
        }
        size_32 += 1 + sz;
        size_64 += 1 + sz;
        for (i = 0; i < sz - 1; i++) marshal_rec(Field(v, i));
        v = Field(v, i);
        goto tailcall;
      }
      }
    }
  }
}

value skt_marshal(		/* ML */
        value obj_v,
	value buf_v,
	value ofs_v,
	value len_v
) {
  int ofs ;
  int len ;
  int ret_len ;

  SKTTRACE;
  ofs = Int_val(ofs_v) ;
  len = Int_val(len_v) ;

  /* Check for alignment.
   */
  if ((len % 4) != 0)
    invalid_argument("marshal:length not on word boundary") ;
  if ((ofs % 4) != 0)
    invalid_argument("marshal:offset not on word boundary") ;

  /* Initialize the extern pointers.
   */
  extern_block = &Byte(buf_v,ofs) ;
  extern_limit = extern_block + len ;
  extern_ptr   = extern_block ;

  /* Initialize the hash table.
   */
  alloc_extern_table();
  obj_counter = 0;
  size_32 = 0;
  size_64 = 0;
  error_occurred = 0 ;

  /* Set aside space for the sizes.
   */
  extern_ptr += 5*4;

  /* Marshal the object.
   */
  marshal_rec(obj_v);

  /* Free the table of shared objects.
   */
  free_extern_table() ;

  /* Write the sizes */
#ifdef ARCH_SIXTYFOUR
  if (size_32 >= (1L << 32) || size_64 >= (1L << 32)) {
    /* The object is so big its size cannot be written in the header.
       Besides, some of the block sizes or string lengths or shared offsets
       it contains may have overflowed the 32 bits used to write them. */
    invalid_argument("marshal: object too big");
  }
#endif
  /* Calculate number of bytes used.
   */
  ret_len = extern_ptr - extern_block ;

  /* Go back to beginning and write magic number and sizes.
   */
  extern_ptr = extern_block ;
  write32(Intext_magic_number);	/* magic number */
  write32(ret_len - 5*4);	/* size in bytes */
  write32(obj_counter);		/* number of objects */
  write32(size_32);		/* words reqd for 32-bit hosts */
  write32(size_64);		/* words reqd for 64-bit hosts */

  /* Round up to next multiple of 4.  Note that this may be
   * larger than the number written above.  Also, we are filling
   * the extra space with 0's.
   */
  for (; ret_len & 3; ret_len++) {
    if (ret_len >= len)
      return Val_int(-1) ;
    extern_block[ret_len] = 0 ;
  }

  if (error_occurred)
    return Val_int(-1) ;
  else
    return Val_int(ret_len) ;
}
