(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* USOCKET *)
(* Author: Mark Hayden, 4/97 *)
(**************************************************************)
let name = "USOCKET"
let failwith s = failwith (name^":"^s)
(**************************************************************)
open Printf
(**************************************************************)

type buf = string
type ofs = int
type len = int

type info = Unix.file_descr * Unix.msg_flag list * (Unix.sockaddr array)
type iovec = buf * ofs * len

type handler =
  | Handler0 of (unit -> unit)
  | Handler1

(**************************************************************)

let unix_wrap debug f =
  try f () with
  | Unix.Unix_error(err,s1,s2) ->
      match err with 
      | Unix.ECONNREFUSED 
      | Unix.ECONNRESET 
      | Unix.EPIPE 
      |	Unix.ENETUNREACH
      | Unix.EAGAIN -> 0
      |	_ ->
	  eprintf "USOCKET:%s:%s\n" debug (Unix.error_message err) ;
	  flush stderr ;
	  raise (Unix.Unix_error(err,s1,s2))

(**************************************************************)

let flatten il = 
  let n = Array.length il in
  let total = Pervasives.ref 0 in
  for i = 0 to pred n do
    let (_,_,len) = il.(i) in
    total := !total + len
  done ;

  let dbuf = String.create !total in
  let dofs = ref 0 in
  for i = 0 to pred n do
    let (buf,ofs,len) = il.(i) in
    String.blit buf ofs dbuf !dofs len ;
    dofs := !dofs + len
  done ;
  dbuf

(**************************************************************)

let fork = Unix.fork
let send s b o l = Unix.send s b o l []
let stdin () = Unix.stdin
let gettimeofday a = a.(0) <- Unix.gettimeofday ()
let socket_of_fd s = s
let read = Unix.read
let static_string = String.create
let static_string_free _ = ()
let preprocess s f a = (s,f,a)
let has_ip_multicast () = false
let setsock_multicast _ _ = failwith "setsock_multicast"
let setsock_join _ _ = failwith "setsock_join"
let setsock_leave _ _ = failwith "setsock_leave"
let setsock_sendbuf _ _ = failwith "setsock_sendbuf"
let setsock_recvbuf _ _ = failwith "setsock_recvbuf"
external int_of_file_descr : Unix.file_descr -> int = "%identity"

(**************************************************************)
(* From util.ml *)
let word_len = 4
let mask1 = pred word_len
let mask2 = lnot mask1
let ceil_word i = (i + mask1) land mask2
(**************************************************************)

let push_int s ofs i =
  if ofs <> ceil_word ofs then
    failwith "push_int:not aligned" ;
  let b3 = (i lsr  0) land 255
  and b2 = (i lsr  8) land 255
  and b1 = (i lsr 16) land 255
  and b0 = (i lsr 24) land 255
  in

  s.[ofs + 0] <- Char.chr b0 ;
  s.[ofs + 1] <- Char.chr b1 ;
  s.[ofs + 2] <- Char.chr b2 ;
  s.[ofs + 3] <- Char.chr b3

let pop_int s ofs =
  if ofs <> ceil_word ofs then
    failwith "pop_int:not aligned" ;
  let b0 = Char.code s.[ofs + 0]
  and b1 = Char.code s.[ofs + 1]
  and b2 = Char.code s.[ofs + 2]
  and b3 = Char.code s.[ofs + 3]
  in

  let i =
    (b3 lsl  0) lor
    (b2 lsl  8) lor
    (b1 lsl 16) lor
    (b0 lsl 24)
  in i

(**************************************************************)

let marshal obj buf ofs len =
  let rep = Obj.marshal obj in
  let rep_len = String.length rep in 
(*
  printf "USOCKET:marshal:%d bytes\n" (rep_len) ;
*)
  let ret_len = ceil_word rep_len in
  if ret_len > len then (
    -1
  ) else (
    (* Copy over and fill padded area with zeroes.
     *)
    String.blit rep 0 buf ofs rep_len ;
    String.fill buf (ofs + rep_len) (ret_len - rep_len) (Char.chr 0) ;
    ret_len
  )

(**************************************************************)

let sendopt (s,f,a) b o l = 
  for i = 0 to pred (Array.length a) do
    unix_wrap "sendopt" (fun () -> 
      Unix.sendto s b o l f a.(i)
    )
  done

let sendvopt info iov = 
  let s = flatten iov in
  sendopt info s 0 (String.length s)

(**************************************************************)

let select a b c d =
  let rec loop () =
    try Unix.select a b c d with
    | Unix.Unix_error(Unix.EINTR,_,_) -> 
	eprintf "USOCKET:block:select:ignoring EINTR error\n" ;
	loop ()
    | Unix.Unix_error(err,s1,s2) ->
      	eprintf "USOCKET:block:select:%s\n" (Unix.error_message err) ;
        failwith "error calling select"
  in loop ()

let recv s b o l = unix_wrap "recv" (fun () -> Unix.recv s b o l [])

(**************************************************************)

type manager = (Unix.file_descr * handler) list * int * (int-> len -> unit)

let manager info max_len upcall =
  (info,max_len,upcall)

let poll (info,max_len,upcall) buf ofs =
  let socks = List.map fst info in
  let avail,_,_ = select socks [] [] 0.0 in
  if avail = [] then false else (
    let sock = List.hd avail in
    begin
      match List.assq sock info with
      | Handler0(f) -> f ()
      | Handler1 ->
 	  let len = recv sock buf ofs max_len in
	  let dint = pop_int buf ofs in
	  upcall dint len
    end ;
    true
  )

(**************************************************************)
(* These are not supported at all.
 *)

type process_handle
let process_socket _ = failwith "process_socket"
let spawn_process _ _ _ = failwith "spawn_process"
let wait_process _ = failwith "wait_proces"
let terminate_process _ = failwith "terminate_proces"

(**************************************************************)
