(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(****************************************************************)
(* SOCKIO.MLI							*)
(* Author: Robbert vanRenesse					*)
(****************************************************************)
(*INCLUDE:open Rpc
 *)

exception Eof

(* This is the type of a TCP address, e.g. ("snotra.cs.cornell.edu", 1234)
 *)
type location = string * int

(* This is the type of requests to the White Pages service.
 *)
type wp_req =
  | WP_UPDATE of string * location list
  | WP_LOOKUP of string

(* Each client is uniquely identified by a number.
 *)
type client_id = int

(* Each message (or message pair in the case of RPC) is uniquely
 * identified by the client identifier, and a sequence number
 * generated by the client.
 *)
type message_id = client_id * int

(* A message consists of the message id and its object identifier.
 *)
type message = message_id * Obj.t

(* A request message is just a message.
 *)
type request = message

(* Responses come in two flavors:  an ordinary (successful) response,
 * or an exceptional error response.
 *)
type response =
  | RESPONSE of message
  | EXCEPTION of message

(* The server is handed a request identifier, which consists of the
 * socket on which the message was received, and the message identifier
 * of the message.
 *)
type request_id

(* When you create a server, you get a ``handle'' back.
 *)
type handle

(* For RPC servers, the handle is extended with a queue of pending
 * requests.
 *)
type rpc_handle

(* For debugging.
 *)
val dump : string -> int -> int -> unit

(* This function may be used to specify the functions to be used
 * for registering and unregistering sockets.
 *)
val register_register :
  (Hsys.socket -> (Hsys.socket -> unit) -> (Hsys.socket -> unit) -> unit) ->
  (Hsys.socket -> unit) -> unit

(* These invoke the functions specified by register_register.  You
 * specify a socket, and the upcalls to be invoked when there's input
 * and when there's a failure.  Most programs will use sockio_register()
 * or ensemble_register() to do this.
 *)
val register :
  Hsys.socket -> (Hsys.socket -> unit) -> (Hsys.socket -> unit) -> unit
val unregister : Hsys.socket -> unit

(* Programs that do not make use of Ensemble should invoke sockio_register
 * for initialization.
 *)
val sockio_register : unit -> unit

(* Programs that use Ensemble should call ensemble_register for initialization.
 *)
val ensemble_register :
  (Hsys.socket -> Hsys.handler -> unit) -> (Hsys.socket -> unit) -> unit

(* Given a request identifier, extract its message identifier.
 *)
val mid_of_rid : request_id -> message_id

(* Given a message identifier, generate a printable string that
 * describes it.
 *)
val string_of_mid : message_id -> string

(* Generate a client of a server.  The server is named, as well as
 * a function that should be invoked when the connection is established.
 * This function gets as arguments a function to send a message, and
 * a function to close the connection.  It has to return a function to
 * deliver messages, and a function to be invoked if the connection
 * breaks.  The last one may return false if you would like to
 * automatically reestablish the connection.  Client() returns a handle.
 *)
val client :
  string ->
  (('a -> unit) -> (unit -> unit) -> ('b -> unit) * (unit -> bool)) ->
  handle

(* Generate a client of an RPC server.  The server is named, and an
 * rpc_handle is returned.
 *)
val bind_to_service : string -> rpc_handle

(* In case you do not want your RPC server to be looked up in the
 * White Pages, you can specify one or more TCP addresses with this
 * function.
 *)
val add_to_service : rpc_handle -> string -> int -> unit

(* Do an RPC.  Specify the RPC server, the message, and the callbacks
 * for normal and exceptional responses.
 *)
val rpc :
  rpc_handle ->
  'a -> (Obj.t -> unit) -> (Obj.t -> unit) -> unit

(* Respond to an RPC request.  Specify the request identifier and
 * the response message.
 *)
val response : request_id -> 'a -> unit

(* Produce an exceptional response to a request.  Specify the request
 * identifier and the exceptional response.
 *)
val except : request_id -> 'a -> unit

(* A debugging routine.
 *)
val debug : unit -> unit

(* If you don't invoke the local select(), you need to call sweep
 * occasionally to keep things rolling along.  Once a second is
 * about enough.
 *)
val sweep : unit -> unit

(* Wait for input on the registered sockets.  Do not wait longer
 * that the given timeout period in seconds.
 *)
val select : float -> unit

(* Create a server of the given name, and on the given port (specify
 * 0 if you don't care.  The function is much like the function
 * specified in client(), except that you cannot request to
 * automatically reestablish connections.
 *)
val server :
  string ->
  Hsys.port ->
  (('a -> unit) -> (unit -> unit) -> ('b -> unit) * (unit -> unit)) ->
  unit

(* Start an RPC server without registering it with the White pages.
 * Specify the port and the upcall that accepts the requests.
 *)
val start : Hsys.port -> (request_id -> 'a -> unit) -> unit

(* Start an RPC server and register it with the White Pages service.
 *)
val start_server :
  string -> Hsys.port -> (request_id -> 'a -> unit) -> unit
