/**************************************************************/
/*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 */
/**************************************************************/
// $Header: /usr/u/hayden/cvs/ensemble/maestro/Maestro_Types.h,v 1.6 1997/05/25 15:41:13 hayden Exp $

#ifndef __MAESTRO_TYPES_H__
#define __MAESTRO_TYPES_H__

/********************************************************************
 *                                                                  *
 * Author:  Alexey Vaysburd  December 96                            *
 *                                                                  *
 * Contents:  Definition of Maestro Wrapper Classes                 *
 *                                                                  *
 ********************************************************************/


#include <assert.h>
#include <iostream.h>
#include <string.h>
#include <malloc.h>

extern "C" {
#include "hot_sys.h"
#include "hot_error.h"
#include "hot_thread.h"
#include "hot_msg.h"
#include "hot_ens.h"
}

#include "Maestro_Config.h"

class Maestro_ErrorHandler;
class Maestro_Base;

class Maestro_Thread;
class Maestro_Semaphore;
class Maestro_Lock;

class Maestro_Message;
class Maestro_EndpID;
class Maestro_EndpList;
class Maestro_String;

class Maestro_GroupMember;

#include "Maestro_OrderedSet.h"

#define check_err(s) if (err != HOT_OK) error->panic(err, s) 

// printing debugging messages
#ifdef MAESTRO_DEBUG
#define putd(s) (cout << (s) << endl) 
#else
#define putd(s)
#endif

// magic numbers for message typechecking for various datatypes
const int MAESTRO_MESSAGE_MAGIC_BASE       = 498752;
const int MAESTRO_MESSAGE_MAGIC_BUFFER     = 269385;
const int MAESTRO_MESSAGE_MAGIC_INT        = 153274;
const int MAESTRO_MESSAGE_MAGIC_ENDP_ID    = 807649;
const int MAESTRO_MESSAGE_MAGIC_ENDP_LIST  = 340127;
const int MAESTRO_MESSAGE_MAGIC_STRING     = 935810;
const int MAESTRO_MESSAGE_MAGIC_MESSAGE    = 693847;

/***************************** Maestro_ErrorHandler *******************/

// This class defines panic() methods that are invoked when an error occurs.
class Maestro_ErrorHandler {
public:
  Maestro_ErrorHandler() {}
  Maestro_ErrorHandler(Maestro_ErrorHandler& heh); 
  virtual ~Maestro_ErrorHandler() {}
  virtual Maestro_ErrorHandler& operator = (Maestro_ErrorHandler& heh);
  virtual void panic(char *context = "unknown problem");
  virtual void panic(hot_err_t err, char *context = "unknown problem");
};

extern Maestro_ErrorHandler Maestro_DefaultErrorHandler;


/******************************* Maestro_Base **************************/

// this is the base class for all following Maestro classes. 
class Maestro_Base {
public:

  inline Maestro_Base(Maestro_ErrorHandler *error_handler = 0);
  Maestro_Base(Maestro_Base& hb) {}
  virtual ~Maestro_Base() {}
  virtual Maestro_Base& operator= (Maestro_Base& hb) { return *this; }
  virtual inline void operator<< (Maestro_Message& msg);
  virtual inline void operator>> (Maestro_Message& msg);
  virtual inline void operator>> (ostream& out) const;

  // magic number for message I/O (to be overloaded for finer typechecking)
  virtual int messageMagic() const { return MAESTRO_MESSAGE_MAGIC_BASE; }

  // Marshal/unmarshal the object into/from a string.
  virtual inline void operator << (Maestro_String& msg);
  virtual inline void operator >> (Maestro_String& msg);

protected:
  Maestro_ErrorHandler *error;
  hot_err_t err;
};

Maestro_Message& operator<< (Maestro_Message& msg, Maestro_Base& hb);
Maestro_Message& operator>> (Maestro_Message& msg, Maestro_Base& hb);
ostream& operator<< (ostream& out, Maestro_Base& hb);

// all options classes inherit from Maestro_Options
struct Maestro_Options : private Maestro_Base {
  virtual void reset() {}
};


/**************************** Maestro_Thread *****************************/

struct Maestro_ThreadOps {
  int stackSize;
};

class Maestro_Thread : virtual public Maestro_Base {
public:

  static inline void create(void (*proc)(void*), void *arg,
			    Maestro_ThreadOps *ops = 0,
			    Maestro_ErrorHandler *error_handler = 0); 
  static inline void usleep(int usecs) ;
}; 


/***************************** Maestro_Semaphore **************************/

class Maestro_Semaphore : virtual public Maestro_Base {
public:

  inline Maestro_Semaphore(int initial_value = 0, 
		    Maestro_ErrorHandler *error_handler = 0);
  Maestro_Semaphore(Maestro_Semaphore& hs);
  virtual inline ~Maestro_Semaphore();
  virtual Maestro_Semaphore& operator= (Maestro_Semaphore& hs);
  virtual inline void inc(); 
  virtual inline void dec();

protected:

   hot_sema_t sema;
};


/********************************* Maestro_Lock ****************************/

class Maestro_Lock : virtual public Maestro_Base {
public:
  
  inline Maestro_Lock(Maestro_ErrorHandler *error_handler = 0); 
  Maestro_Lock(Maestro_Lock& hl);
  virtual inline ~Maestro_Lock();
  virtual Maestro_Lock& operator= (Maestro_Lock& hl);
  virtual inline void lock();
  virtual inline void unlock();

protected:

  hot_lock_t lck;
};


/********************************* Maestro_Barrier ***********************/

typedef enum { 
  MAESTRO_BARRIER_OPEN, 
  MAESTRO_BARRIER_CLOSED 
} Maestro_BarrierState;

// if Barrier's state is CLOSED, pass() will block until open() is called 
class Maestro_Barrier : virtual public Maestro_Base {
public:

  inline Maestro_Barrier(Maestro_BarrierState theState = MAESTRO_BARRIER_OPEN, 
		  Maestro_ErrorHandler *error_handler = 0);
  Maestro_Barrier(Maestro_Barrier& hc);
  inline ~Maestro_Barrier();
  Maestro_Barrier& operator= (Maestro_Barrier& hc);
  inline void close();
  inline void open();
  inline void pass(); 
  inline int isOpen() { return (state == MAESTRO_BARRIER_OPEN); }
  inline int isClosed() { return (state == MAESTRO_BARRIER_CLOSED); }

protected:
 
  Maestro_Semaphore *sema;
  Maestro_Lock *mutex;
  int nwaiting;
  Maestro_BarrierState state;
};


/**************************** Maestro_String ********************************/

struct Maestro_String : virtual public Maestro_Base {
  inline Maestro_String(char *str = 0);
  inline Maestro_String(Maestro_String& str);
  inline virtual ~Maestro_String();
  virtual inline Maestro_String& operator= (Maestro_String&);
  virtual inline Maestro_String& operator= (char*);
  inline int operator== (Maestro_String& str);
  
  inline void operator<< (Maestro_Message &msg);
  inline void operator>> (Maestro_Message &msg);
  inline int messageMagic() const { return MAESTRO_MESSAGE_MAGIC_STRING; }
  inline void operator>> (ostream &out) const;

  char *s;
  int size;
};


/**************************** Maestro_Message ******************************/

// This mirrors the corresp. definition in hot_msg.h
typedef enum {
  MAESTRO_MSG_SEEK_FROM_HEAD,
  MAESTRO_MSG_SEEK_FROM_TAIL,
  MAESTRO_MSG_SEEK_FROM_CURRENT
} Maestro_MsgSeek;

class Maestro_Message : virtual public Maestro_Base {
friend Maestro_GroupMember;

public:

  inline Maestro_Message(Maestro_ErrorHandler *error_handler = 0);

  inline Maestro_Message(Maestro_Message& msg, 
		  Maestro_ErrorHandler *error_handler = 0);

  inline Maestro_Message(hot_msg_t hmsg, 
		  Maestro_ErrorHandler *error_handler = 0);

  inline Maestro_Message(void *data,
		  unsigned nbytes,
		  void (*on_release)(void*) = 0, 
		  void *env = 0,
		  Maestro_ErrorHandler *error_handler = 0);

  virtual inline ~Maestro_Message();
  virtual inline Maestro_Message& operator= (Maestro_Message &msg);
  virtual inline Maestro_Message& operator<<= (Maestro_Message &msg);

  virtual inline void read(int &i);
  
  virtual inline Maestro_Message& operator>> (int &i) { 
    read(i); return *this; 
  }
  
  virtual inline void read(void *buf, unsigned size);
  virtual inline void look(void *&data, unsigned size);

  virtual inline void write(int i);
  
  virtual inline Maestro_Message& operator<< (int i) { 
    write(i); return *this; 
  }
  
  virtual inline void write(void *buf, int size);

  virtual inline unsigned size();
  virtual inline unsigned getPos();
  virtual inline void setPos(unsigned offset);
  virtual inline void seek(int offset, Maestro_MsgSeek whence);

  virtual inline void pack(Maestro_Message *msgs, unsigned nmsgs);
  virtual inline void unpack(Maestro_Message *&msgs, unsigned &nmsgs);
  
  virtual inline void reset();
  virtual inline int operator== (Maestro_Message &hm) { return 0; } 
  virtual inline int isWritable();

  void inline operator<< (Maestro_Message &msg);
  void inline operator>> (Maestro_Message &msg);
  int inline messageMagic() const { return MAESTRO_MESSAGE_MAGIC_MESSAGE; }  

  void setName(char *s) { name = s; }

protected:
  hot_msg_t msg;
  Maestro_String name;
}; 


/**************************** Maestro_Buffer *******************************/

struct Maestro_Buffer : virtual public Maestro_Base {
  
  inline Maestro_Buffer(void *theBuf, int theSize) {
    buf = theBuf;
    size = theSize;
  }

  inline void operator << (Maestro_Message &msg) { msg.read(buf, size); }
  inline void operator >> (Maestro_Message &msg) { msg.write(buf, size); }
  
  void *buf;
  int size;
};

/**************************** Maestro_EndpID *************************/

class Maestro_EndpID : virtual public Maestro_Base {
friend Maestro_GroupMember;

public:

  inline Maestro_EndpID(Maestro_ErrorHandler *error_handler = 0); 
  inline Maestro_EndpID(hot_endpt_t ep, 
			Maestro_ErrorHandler *error_handler = 0);
  
  virtual inline int operator== (Maestro_EndpID &eid);
  virtual inline int operator!= (Maestro_EndpID &eid);
  virtual inline Maestro_EndpID& operator= (Maestro_EndpID &eid);
  virtual inline Maestro_EndpID& operator= (hot_endpt_t &ep); 

  // Message I/O methods.
  inline void operator<< (Maestro_Message &msg);
  inline void operator>> (Maestro_Message &msg);
  inline int messageMagic() const { return MAESTRO_MESSAGE_MAGIC_ENDP_ID; }

  inline void operator>> (ostream &out) const { cout << endpID.name << endl; }

protected:

  hot_endpt_t endpID; // defined as struct {char name[HOT_ENDP_MAX_NAME_SIZE];} 
};


/*************************** Maestro_EndpList *****************************/

typedef Maestro_OrderedSet(Maestro_EndpID) Maestro_EndpList_Base;

class Maestro_EndpList : virtual public Maestro_EndpList_Base {
public:
			   
  inline Maestro_EndpList(Maestro_ErrorHandler *error_handler = 0);

  inline Maestro_EndpList(Maestro_EndpID& eid, 
			  Maestro_ErrorHandler *error_handler = 0);

  inline Maestro_EndpList(Maestro_EndpList& ep_list, 
			  Maestro_ErrorHandler *error_handler = 0);

  inline Maestro_EndpList(hot_endpt_t endps[], unsigned nendp, 
			  Maestro_ErrorHandler *error_handler = 0);

  inline int messageMagic() const { return MAESTRO_MESSAGE_MAGIC_ENDP_LIST; }
};

typedef Maestro_OrderedSet(Maestro_Message) Maestro_MessageList;

#endif // __MAESTRO_TYPES_H__
