(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* LEAVE.ML : leave protocol *)
(* Author: Mark Hayden, 12/95 *)
(**************************************************************)
open Event
open Layer
open Util
open View
(**************************************************************)
let name = Trace.source_file "LEAVE"
let failwith s = failwith (Util.failmsg name s)
(**************************************************************)

type header = Leave | Exit

type state = {
  wait_stable : bool ;
  mutable exited : bool ;
  mutable got_view : bool ;
  mutable leaving : bool ;
  mutable elected : bool
}

let dump (ls,vs) s =
  eprintf "LEAVE:dump:%s\n" ls.name ;
  eprintf "  exited=%b, got_view=%b elected=%b leaving=%b\n" 
    s.exited s.got_view s.elected s.leaving

let init () (ls,vs) = {
  wait_stable = Param.bool vs.params "leave_wait_stable" ;
  exited      = false ;
  got_view    = false ;
  leaving     = false ;
  elected     = ls.am_coord
}

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let ack = make_acker name dnnm in
  let up_hdlr ev abv () = up ev abv
  and uplm_hdlr ev hdr = match getType ev,hdr with
  | ECast, Leave ->
      let origin = getOrigin ev in 
      if origin <> ls.rank then
	dnnm (suspectReason name [origin] name) ;
      ack ev ; free name ev

  | ECast, Exit ->
      let origin = getOrigin ev in
      if s.got_view			(* I've already got the new view *)
      && origin <> ls.rank		(* Not from me *)
      && not s.exited then (		(* I've not gotten an exit yet *)
	s.exited <- true ;
	dnnm (create name EExit[])
      ) ;
      ack ev ; free name ev

  | _ -> failwith "unknown local message"

  and upnm_hdlr ev = match getType ev with
  | EView ->
      (* Remember that a view arrived.
       *)
      s.got_view <- true ;
      
      (* Only pass up views if we are not leaving.
       *)
      if not s.leaving then
      	upnm ev ;

      (* Singleton groups may not get any stability events.
       *)
      if ((ls.nmembers = 1) || (not s.wait_stable))
      && (not s.exited) then (
	 s.exited <- true ;
	 dnnm (create name EExit[])
      )

  | EStable ->
      (* We use stability messages to find a good time to GC
       * this protocol stack.  
       *)
      if s.wait_stable
      && s.got_view 
      && (not s.exited)
      then (
	let mins = getStability ev in
	let maxs = getNumCasts ev in
	if mins = maxs then (
	  (* Exit myself.
	   *)
	  s.exited <- true ;
	  dnnm (create name EExit[]) ;

	  (* Decide whether to tell everyone else.
	   *)
	  if s.elected			(* I'm elected *)
	  && ls.nmembers > 1		(* And not singleton view *)
	  then
	    dnlm (create name ECast[]) Exit ;
	)
      ) ;
      upnm ev

  | EElect ->
      s.elected <- true ;
      upnm ev

  | ELeave ->
      (* Cast a message to the group.
       *)
      dnlm (castEv name) Leave ;

      (* Mark me as leaving.
       *)
      s.leaving <- true ;	
      upnm ev

  | EDump -> ( dump (ls,vs) s ; upnm ev )
  | _ -> upnm ev

  and dn_hdlr ev abv = dn ev abv ()
  and dnnm_hdlr ev = match getType ev with

    (* Don't exit twice, and remember if someone
     * else generates an exit.
     *)
  | EExit ->
      if s.exited then (
	free name ev
      ) else (
	s.exited <- true ;
	dnnm ev ;
      )

    (* Why necessary? *)
  | EBlockOk ->
      if s.exited then
        free name ev
      else
        dnnm ev

  | _ -> dnnm ev

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vf = Layer.hdr init hdlrs None NoOpt args vf

let _ = 
  Param.default "leave_wait_stable" (Param.Bool true) ;
  Layer.install name (Layer.init l)

(**************************************************************)
