(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* INTRA.ML *)
(* Author: Mark Hayden, 10/96 *)
(**************************************************************)
open Trans
open Util
open Layer
open Event
open View
(**************************************************************)
let name = Trace.source_file "INTRA"
(**************************************************************)

type header =
  | View of View.state
  | Fail of rank list

type state = {
  mutable elected 	: Once.t ;	(* am I the coordinator? *)
  mutable new_view 	: Once.t ;	(* have I done a EView? *)
  mutable failed 	: rank list 	(* failed members *)
}

let dump (ls,vs) s =
  eprintf "INTRA:dump:%s\n" ls.name ;
  eprintf "  new_view=%b elected=%b failed=%s\n" 
    (Once.isset s.new_view) (Once.isset s.elected)
    (string_of_int_list s.failed)

let init () (ls,vs) = {
  elected   	= Once.create "elected" ;
  new_view      = Once.create "new_view" ;
  failed    	= []
}

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let failwith m = dump (ls,vs) s ; failwith (name^":"^m) in
  let assert = make_assert failwith in
  let ack = make_acker name dnnm in
  let log = Trace.log name ls.name in

  let up_hdlr ev abv () = up ev abv
  and uplm_hdlr ev hdr = match getType ev,hdr with

  (* New view arrived.  If not from me, check it out.  If
   * accepted, bounce off bottom.  
   *)
  | ECast, View(new_vs) -> (
      let origin = getOrigin ev in
      if origin <> ls.rank then (
	if Once.isset s.elected		(* I'm coordinator *)
        || Once.isset s.new_view	(* I've already accepted a view *)
	|| ls.rank < origin		(* my rank is lower *)
	|| List.mem origin s.failed	(* coord is failed *)
	|| not (array_mem ls.endpt new_vs.view) (* I am not in the new view *)
  (*    || subtract remaining view' <> [] (* some members are missing *)*)
	then (
	  log (fun () -> sprintf "View:rejected:%s" (View.to_string new_vs.view)) ;
	  dnnm (suspectReason name [origin] name) ;
	) else (
	  log (fun () -> sprintf "View:accepted:%s" (View.to_string new_vs.view)) ;
	  Once.set s.new_view ;
	  dnnm (create name EView[ViewState new_vs])
	)
      ) ;
      ack ev ; free name ev
    )

  (* New failure announced.  If not from me, check it out.
   * If accepted, bounce off bottom. 
   *)
  | ECast, Fail(failures) ->
      let origin = getOrigin ev in
      if origin <> ls.rank then (
        let higher = Util.sequence origin in
	if Once.isset s.elected		(* I've been elected *)
	|| ls.rank < origin		(* my rank is lower *)
	|| not (Lset.super failures higher) (* failures don't include all lower ranked members *)
	|| not (Lset.super failures s.failed) (* he doesn't include failures I've seen *)
	|| List.mem origin s.failed	(* coord is failed *)
	|| List.mem origin failures	(* he is failing himself *)
	|| List.mem ls.rank failures	(* I am being failed *)
	then (
	  log (fun () -> sprintf "Fail:rejected:%s" (string_of_int_list failures)) ;
	  dnnm (create name ESuspect[
	    Suspects [origin] ;
	    SuspectReason name
	  ])
	) else (
	  log (fun () -> sprintf "Fail:accepted:%s" (string_of_int_list failures)) ;
	  s.failed <- failures ;
	  dnnm (create name EFail[(Failures failures)])
	)
      ) ;
      ack ev ; free name ev
  | _ -> failwith "unknown local message"

  and upnm_hdlr ev = match getType ev with
  (* EElect: I've been elected.
   *)
  | EElect ->
      Once.set s.elected ;
      upnm ev

  | EDump -> ( dump (ls,vs) s ; upnm ev )
  | _ -> upnm ev

  and dn_hdlr ev abv = dn ev abv ()

  and dnnm_hdlr ev = match getType ev with
  (* EView: send out view and bounce locally.
   *)
  | EView ->
      let vs = getViewState ev in
      if Once.isset s.new_view then (
	log (fun () -> "dropping EView because view is already accepted") ;
	free name ev
      ) else (
      	Once.set s.new_view ;
      	log (fun () -> sprintf "EView:%s" (View.to_string vs.view)) ;
      	assert (Once.isset s.elected) "EView when not elected" ;
      	dnlm (castEv name) (View vs) ;
      	dnnm ev
      )

  (* EFail: send out failures and bounce locally.
   *)
  | EFail ->
      let failures = getFailures ev in
      log (fun () -> sprintf "EFail:%s" (string_of_int_list failures)) ;
      assert (Once.isset s.elected) "EFail when not elected" ;
      s.failed <- failures ;
      dnlm (castEv name) (Fail(s.failed)) ;
      dnnm ev

  | _ -> dnnm ev

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vf = Layer.hdr init hdlrs None NoOpt args vf

let _ = Layer.install name (Layer.init l)

(**************************************************************)
