(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(* INTER.ML *)
(* Author: Mark Hayden, 10/96 *)
(**************************************************************)
open Util
open Layer
open View
open Event
open Trans
(**************************************************************)
let name = Trace.source_file "INTER"
(**************************************************************)

type header = NoHdr
  | Request of Endpt.full * View.state
  | Granted of View.state
  | Denied  of View.state
	
type states =
  | Normal
  | Merging
  | FailedMerge
  | InstalledView

type state = {
  mutable state		: states ;	(* automata state *)
  mutable blocked	: bool ;	(* have I seen an EBlockOk? *)
  mutable max_ltime 	: ltime ;	(* max ltime I've seen on merge requests *)
  mutable endpts_seen 	: Endpt.id list ; (* endpts I've seen anywhere *)
  mutable merger_info   : (Endpt.full * View.id) list (* view_ids merging with me *)
}

let string_of_state = function
  | Normal  	  -> "Normal"
  | Merging 	  -> "Merging"
  | FailedMerge   -> "FailedMerge"
  | InstalledView -> "InstalledView"

let dump (ls,vs) s =
  eprintf "INTER:dump:%s %s\n" ls.name (string_of_state s.state) ;
  eprintf "  am_coord=%b, blocked=%b\n" ls.am_coord s.blocked ;
  eprintf "  view_id=%s, max_ltime=%d\n" (View.string_of_id vs.view_id) s.max_ltime ;
  eprintf "  view=%s\n" (View.to_string vs.view) ;
  eprintf "  endpts_seen=%s\n" (Endpt.string_of_id_list s.endpts_seen)

let init () (ls,vs) = {
  endpts_seen  	= Array.to_list vs.view ;
  state		= Normal ;
  blocked	= false ;
  max_ltime  	= fst vs.view_id ;
  merger_info   = []
}


let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let failwith m = dump (ls,vs) s ; failwith (name^":"^m) in
  let log        = Trace.log name ls.name in
  let assert 	 = make_assert failwith in
  let assertions = List.iter (fun (v,s) -> if not v then failwith s) in
  let ack 	 = make_acker name dnnm in

  let up_hdlr ev abv hdr = match getType ev,hdr with
    (* EMergeRequest (Coordinator): merge request from remote
     * coordinator.
     *)
  | EMergeRequest, Request(merge_con,merge_vs) ->
      let merge_view = Array.to_list merge_vs.view in
      let merge_view = Lset.sort merge_view in

      if ls.am_coord			(* I'm coord *)
      && not s.blocked			(* haven't blocked yet *)
      && Lset.disjoint merge_view s.endpts_seen (* no endpt that I've seen before *)
      then (
      	(* Accept request.  Update logical time, add to
	 * lists of possibly previous view_ids and the endpts
	 * we've seen so far.  Then send on up.  
      	 *)
	log (fun () -> sprintf "EMergeRequest:accepted:%s" (View.to_string merge_vs.view)) ;
        s.max_ltime        <- max s.max_ltime (fst merge_vs.view_id) ;
	s.endpts_seen      <- Lset.union s.endpts_seen merge_view ;
        s.merger_info      <- (merge_con,merge_vs.view_id) :: s.merger_info ;
	up (set name ev [Mergers merge_vs]) abv
      ) else (
      	(* Deny request.
	 *)
      	let reason_dbg =
	  (if not ls.am_coord then ":not_coord" else "") ^
	  (if s.blocked then ":blocked" else "") ^
	  (if Lset.disjoint merge_view s.endpts_seen then "" else ":seen_before")
	in
	log (fun () -> sprintf "EMergeRequest:rejected:%s%s" (View.to_string merge_vs.view) reason_dbg) ;

      	dnlm (create name EMergeDenied[
      	  Contact(merge_con,Some merge_vs.view_id)
      	]) (Denied merge_vs) ;
      	free name ev
      )

  | _, NoHdr -> up ev abv
  | _ -> failwith "bad up event"

  and uplm_hdlr ev hdr = match getType ev,hdr with

    (* EMergeGranted(Coordinator): Remote merging coordinator: new view arrived
     * from coordinator.  bounce off bottom.
     *)
  | EMergeGranted, Granted(new_vs) ->
      if ls.am_coord
      && s.state = Merging 
      && List.mem vs.view_id new_vs.prev_ids
      (* BUG: also check that correct members are in the view. *)
      then (
      	log (fun () -> "EMergeGranted:accepted") ;
        (* Next shouldn't be a bug (??) *)
	assert (fst new_vs.view_id > fst vs.view_id) "install view:non-increasing ltime" ;
	assert (snd new_vs.view_id = new_vs.view.(0)) "contact of view did not make view_id" ;

	(* Install the view locally.
	 *)
	s.state <- InstalledView ;
	dnnm (create name EView[ViewState new_vs])
      ) else (
      	log (fun () -> "EMergeGranted:rejected")
      ) ;
      free name ev

    (* EMergeDenied: if the view_id matches my view_id,
     * then pass up an EMergeFailed event.
     *)
  | EMergeDenied, Denied(merge_vs) ->
      if ls.am_coord
      && s.state = Merging 
      && vs = merge_vs
      then (
        log (fun () -> "EMergeDenied(this view)") ;
      	s.state <- FailedMerge ;
	upnm (create name EMergeFailed[])
      ) ;
      free name ev

  | _ -> failwith "unknown local message"

  and upnm_hdlr ev = match getType ev with
  | EView ->
      s.state <- InstalledView ;
      upnm ev

    (* EBlockOk: mark state and pass on.
     *)
  | EBlockOk -> 
      s.blocked <- true ; 
      upnm ev

    (* EMergeFailed: my EMerge timed out.  If the merge
     * was still in progress, send it on up.
     *)
  | EMergeFailed ->
      if ls.am_coord && s.state = Merging then (
        log (fun () -> "EMergeFailed") ;
      	s.state <- FailedMerge ;
      	upnm ev
      ) else (
      	free name ev
      )

  | EDump -> ( dump (ls,vs) s ; upnm ev )
  | _ -> upnm ev

  and dn_hdlr ev abv = match getType ev with
    (* EMergeRequest. Sends merge request.  Note that EMerge
     * should only be done after zero failures and by the
     * original coordinator of the group.  
     *)
  | EMergeRequest ->
      (* Check a bunch of assertions. *)
      let contact,contact_vid = getContact ev in
      assertions [
	ls.am_coord,		   	"non-coord merging" ;
	not (array_mem (fst contact) vs.view),
      	       	                        "merging with endpt already in group" ;
	s.state = Normal,	   	"bad merge state"
      ] ;

      log (fun () -> sprintf "EMergeRequest:with:%s" (Endpt.string_of_full contact)) ;
      s.state <- Merging ;
      let contact = (ls.endpt,ls.addr) in
      dn ev abv (Request(contact,vs))

  | _   -> dn ev abv NoHdr

  and dnnm_hdlr ev = match getType ev with
    (* EView: Forward view to merging groups.
     *)
  | EView ->
      assert (s.state <> InstalledView) "EView after view accepted" ;

      (* Calculate the new view id and add to event.
       *)
      s.state <- InstalledView ;
      let merger_view_ids = List.map snd s.merger_info in
      let new_view_id = (succ s.max_ltime,ls.endpt) in
      let new_vs = getViewState ev in
      let new_vs = View.set new_vs [
	Vs_view_id new_view_id ;
	Vs_prev_ids (vs.view_id::merger_view_ids)
      ]	in

      (* Send view state to merging coordinators (who will forward
       * it for me).  
       *)
      log (fun () -> sprintf "EView:view:%s" (View.to_string new_vs.view)) ;
      List.iter (fun (con,vid)  ->
        log (fun () -> sprintf "EView:with:%s" (Endpt.string_of_full  con)) ;
        dnlm (create name EMergeGranted[
          Contact(con,Some vid)
        ]) (Granted new_vs)
      ) s.merger_info ;

      (* Pass view to my partition.
       *)
      dnnm (set name ev [ViewState new_vs])

  | _ -> dnnm ev

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vf = Layer.hdr init hdlrs None NoOpt args vf

let _ = Layer.install name (Layer.init l)

(**************************************************************)
