(**************************************************************)
(*
 *  Ensemble, (Version 0.40)
 *  Copyright 1997 Cornell University
 *  All rights reserved.
 *
 *  See ensemble/doc/license.txt for further information.
 *)
(**************************************************************)
(**************************************************************)
(*
 *  TransisE, (Version 0)
 *  Hebrew University
 *  Copyright Notice
 *
 *  The contents of this file are subject to copyright by
 *  the Hebrew University, which reserves all rights.  Use,
 *  distribution and copying of this material is expressly
 *  prohibited except by prior written permission from
 *  the Hebrew University or from its appropriately authorized 
 *  agents and licensors.
 *)
(**************************************************************)
(* TSYNC.ML : Extended Virtual Synchrony *)
(* Author: Ohad Rodeh, 12/96 *)
(* Based on code by: Mark Hayden  - SYNC*)
(**************************************************************)
(*
TSYNC - Extended Virtual Synchrony
----------------------------------

Coordinator -  ECast(Block)
members     -  Block the sending of new messages, and ECast(BlockOk)
members     -  receives BlockOk from all members, and ESend - UpBlobkOk,
               and continue. 

The blocking of local messages is done via - 
  dnlm EBlock
  bounces as EBlock
  bounces down as EBlockOk

Notes
-----
This layer work with/without self/deliver.
*)
(**************************************************************)
open Event
open Util
open Layer
open Trans
open View
(**************************************************************)
let name = Trace.source_file "TSYNC"
(**************************************************************)

type header =
| NoHdr
| Block
| BlockOk

type state = {
    failed	          : bool array ;
    block_ok	          : bool array ;
    mutable req_up_block_ok : bool ;
    mutable dn_block        : bool ;
    mutable up_block_ok     : bool ;
    mutable up_block        : bool 
  }

type ('a,'b) t = (header,'a,'b,state) Layer.t

(**************************************************************)

let dump (ls,vs) s =
  eprintf "TSYNC:dump:%s\n" ls.name;
  eprintf "  req_up_block_ok=%b rank=%d\n" s.req_up_block_ok ls.rank ;
  eprintf "  dn_block=%b up_block=%b up_block_ok=%b\n"
    s.dn_block s.up_block s.up_block_ok ;
  eprintf "  failed  =%s\n" (string_of_bool_array s.failed) ;
  eprintf "  block_ok=%s\n" (string_of_bool_array s.block_ok)

(**************************************************************)

let init () (ls,vs) = {
  failed      	= array_create name ls.nmembers false ;
  block_ok    	= array_create name ls.nmembers false ;
  req_up_block_ok  = false;
  dn_block    	= false ;
  up_block_ok	= false ;
  up_block	= false 
}

(**************************************************************)

let hdlrs s (ls,vs) {up_out=up;upnm_out=upnm;dn_out=dn;dnlm_out=dnlm;dnnm_out=dnnm} =
  let failwith m = dump (ls,vs) s ; failwith (name^":"^m) in
  let log = Trace.log name ls.name in
  let assert a s = if not a then failwith s in
  let null_arr = array_create name ls.nmembers 0 in

  let check_ok () =
    if s.dn_block  
      && s.req_up_block_ok 
      && (not s.up_block_ok) 
    then (
      let all_true = ref true in
      for i = 0 to pred (ls.nmembers) do
        if not (s.block_ok.(i) || s.failed.(i)) then
	  all_true := false
      done ;
      if !all_true then (
      	log (fun () -> sprintf "[%d]EBlockOk" ls.rank);
	s.up_block_ok <- true ;
      	upnm (create name EBlockOk[])
      )
    )
  in

  let up_hdlr ev abv hdr = match getType ev, hdr with
  | _,NoHdr -> up ev abv
  | _ -> failwith "bad up event"

  and uplm_hdlr ev hdr = match getType ev,hdr with
  (* Block: cast from coordinator.  If not blocking then bounce
     EBlock off bottom of the group.  If blocking then acknowledge the
     event and do nothing else.  
     *)
  | ECast, Block ->
      if (getOrigin ev) <> ls.rank then (
      	if not s.dn_block then (
	  s.dn_block <- true ;
	  dnnm (create name EBlock[])
	  )
      );
      free name ev

  (* BlockOk: Got block Ok from other members, mark him
   * as OK and check whether we're done blocking.
   *)
  | (ECast|ESend), BlockOk ->
      if (getOrigin ev) <> ls.rank then (
	if s.req_up_block_ok then 
	  log (fun () -> sprintf "[%d]BlockOk %d" ls.rank (getOrigin ev));
      	s.block_ok.((getOrigin ev)) <- true ;
      	check_ok () 
	);
      free name ev
  | _ -> failwith "bad up event"

  and upnm_hdlr ev = match getType ev with
  (* EFail: Mark some members as being failed.  Check if
   * we're done blocking.
   *)
  | EFail ->
      let ranks = getFailures ev in
      List.iter (fun rank ->
        s.failed.(rank) <- true
      ) ranks ;
      upnm ev ;
      check_ok ()
  (* EBlock: EBlock bounced off bottom.  Now bounce off
   * top.
   *)

  | EBlock ->
      assert (not s.up_block) "2nd EBlock" ;
      s.up_block <- true ;
      upnm ev
  | EBlockOk -> failwith "bad EBlockOk"
  | EDump -> (dump (ls,vs) s; upnm ev)
  | _ -> upnm ev

  and dn_hdlr ev abv = dn ev abv NoHdr

  and dnnm_hdlr ev = match getType ev with
  (* 1. Layer above has started blocking group.  If
   * blocking is already done then deliver EBlockOk.
   *)
  | EBlock ->
      assert (not s.req_up_block_ok) "2nd EBlock" ;
      s.req_up_block_ok <- true ;
      log (fun () -> sprintf "[%d]Started Blocking" ls.rank);
      if not s.dn_block then (
        s.dn_block <- true ;
	if ls.nmembers > 1 then dnlm (castEv name) Block;
        dnnm ev				(* pass on *)
      ) else (
      	free name ev
      ) ;
      check_ok () 
  (* 4. EBlock was bounced off top.  Acknowledge
   * Block cast if haven't done so already.
   *)
  | EBlockOk ->
      assert s.up_block "EBlockOk w/o EBlock" ;
      if ls.nmembers > 1 then dnlm (castEv name) BlockOk;
      s.block_ok.(ls.rank) <- true;
      check_ok ();
      free name ev 
  | _ -> dnnm ev

in {up_in=up_hdlr;uplm_in=uplm_hdlr;upnm_in=upnm_hdlr;dn_in=dn_hdlr;dnnm_in=dnnm_hdlr}

let l args vs = Layer.hdr_state init hdlrs None NoOpt args vs

let _ = Layer.install name (Layer.init l)

(**************************************************************)





